<?php
declare(strict_types=1);

/*
|--------------------------------------------------------------------------
| Genel Render
|--------------------------------------------------------------------------
*/
function panel_render(string $page, array $data = []): void {
  $tenant = $GLOBALS['tenant'];
  $u = auth_user();
  view('layouts/panel', array_merge($data, [
    'tenant' => $tenant,
    'user'   => $u,
    'page'   => $page,
  ]));
}

/*
|--------------------------------------------------------------------------
| Dashboard
|--------------------------------------------------------------------------
*/
function panel_dashboard(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  // Customers
  $st = $pdo->prepare("SELECT COUNT(*) c FROM customers WHERE company_id=? AND deleted_at IS NULL");
  $st->execute([$cid]);
  $customersCount = (int)($st->fetch()['c'] ?? 0);

  // Products
  $st = $pdo->prepare("SELECT COUNT(*) c FROM products WHERE company_id=? AND deleted_at IS NULL");
  $st->execute([$cid]);
  $productsCount = (int)($st->fetch()['c'] ?? 0);

  // Open Invoices (draft + sent)
  $st = $pdo->prepare("
    SELECT COUNT(*) c 
    FROM documents 
    WHERE company_id=? 
      AND type='invoice' 
      AND deleted_at IS NULL 
      AND status IN ('draft','sent')
  ");
  $st->execute([$cid]);
  $openInvoices = (int)($st->fetch()['c'] ?? 0);

  // Open Quotes (draft + sent)
  $st = $pdo->prepare("
    SELECT COUNT(*) c 
    FROM documents 
    WHERE company_id=? 
      AND type='quote' 
      AND deleted_at IS NULL 
      AND status IN ('draft','sent')
  ");
  $st->execute([$cid]);
  $openQuotes = (int)($st->fetch()['c'] ?? 0);

  panel_render('dashboard', compact('customersCount','productsCount','openInvoices','openQuotes'));
}


/*
|--------------------------------------------------------------------------
| Customers
|--------------------------------------------------------------------------
*/
function panel_customers(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $st = $pdo->prepare("
    SELECT * FROM customers
    WHERE company_id=? AND deleted_at IS NULL
    ORDER BY id DESC
  ");
  $st->execute([$cid]);
  $rows = $st->fetchAll();

  $ok = flash_get('ok');
  $err = flash_get('err');
  panel_render('customers', compact('rows','ok','err'));
}

function panel_customers_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/customers"));
  }

  $action = (string)($_POST['action'] ?? 'create');

  if ($action === 'create') {
    $name = trim((string)($_POST['name'] ?? ''));
    $email = trim((string)($_POST['email'] ?? ''));
    $phone = trim((string)($_POST['phone'] ?? ''));
    $address = trim((string)($_POST['address'] ?? ''));
    $tax_office = trim((string)($_POST['tax_office'] ?? ''));
    $tax_no = trim((string)($_POST['tax_no'] ?? ''));

    if ($name === '') {
      flash_set('err', 'Müşteri adı zorunlu.');
      redirect(base_path("/{$slug}/customers"));
    }

    $st = $pdo->prepare("
      INSERT INTO customers (company_id,name,email,phone,address,tax_office,tax_no,is_active)
      VALUES (?,?,?,?,?,?,?,1)
    ");
    $st->execute([$cid,$name,$email ?: null,$phone ?: null,$address ?: null,$tax_office ?: null,$tax_no ?: null]);

    flash_set('ok', 'Müşteri eklendi.');
    redirect(base_path("/{$slug}/customers"));
  }

  if ($action === 'trash') {
    $id = (int)($_POST['id'] ?? 0);
    if ($id > 0) {
      $st = $pdo->prepare("UPDATE customers SET deleted_at=NOW() WHERE id=? AND company_id=?");
      $st->execute([$id,$cid]);
      flash_set('ok', 'Müşteri çöp kutusuna taşındı.');
    }
    redirect(base_path("/{$slug}/customers"));
  }

  flash_set('err', 'Bilinmeyen işlem.');
  redirect(base_path("/{$slug}/customers"));
}

/*
|--------------------------------------------------------------------------
| Products
|--------------------------------------------------------------------------
*/
function panel_products(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $st = $pdo->prepare("
    SELECT * FROM products
    WHERE company_id=? AND deleted_at IS NULL
    ORDER BY id DESC
  ");
  $st->execute([$cid]);
  $rows = $st->fetchAll();

  $ok = flash_get('ok');
  $err = flash_get('err');
  panel_render('products', compact('rows','ok','err'));
}

function panel_products_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/products"));
  }

  $action = (string)($_POST['action'] ?? 'create');

  if ($action === 'create') {
    $name = trim((string)($_POST['name'] ?? ''));
    $unit = trim((string)($_POST['unit'] ?? 'adet')) ?: 'adet';
    $price = (float)($_POST['price'] ?? 0);
    $vat = (float)($_POST['vat_rate'] ?? 20);
    $desc = trim((string)($_POST['description'] ?? ''));

    if ($name === '') {
      flash_set('err', 'Ürün/Hizmet adı zorunlu.');
      redirect(base_path("/{$slug}/products"));
    }

    $st = $pdo->prepare("
      INSERT INTO products (company_id,name,description,unit,price,vat_rate,is_active)
      VALUES (?,?,?,?,?,?,1)
    ");
    $st->execute([$cid,$name,$desc ?: null,$unit,$price,$vat]);

    flash_set('ok', 'Ürün/Hizmet eklendi.');
    redirect(base_path("/{$slug}/products"));
  }

  if ($action === 'trash') {
    $id = (int)($_POST['id'] ?? 0);
    if ($id > 0) {
      $st = $pdo->prepare("UPDATE products SET deleted_at=NOW() WHERE id=? AND company_id=?");
      $st->execute([$id,$cid]);
      flash_set('ok', 'Ürün/Hizmet çöp kutusuna taşındı.');
    }
    redirect(base_path("/{$slug}/products"));
  }

  flash_set('err', 'Bilinmeyen işlem.');
  redirect(base_path("/{$slug}/products"));
}

/*
|--------------------------------------------------------------------------
| Settings & Company
|--------------------------------------------------------------------------
*/
function panel_settings(): void { panel_render('settings'); }
function panel_company(): void { panel_render('company'); }

/*
|--------------------------------------------------------------------------
| Documents - Create (GET)
|--------------------------------------------------------------------------
*/
function panel_create(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $customers = $pdo->prepare("SELECT id,name FROM customers WHERE company_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
  $customers->execute([$cid]);
  $customers = $customers->fetchAll();

  // include description so the create form can auto-fill it
  $products = $pdo->prepare("SELECT id,name,description,unit,price,vat_rate FROM products WHERE company_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
  $products->execute([$cid]);
  $products = $products->fetchAll();

  $st = $pdo->prepare("SELECT default_currency, default_currency_symbol, default_vat_rate FROM company_settings WHERE company_id=? LIMIT 1");
  $st->execute([$cid]);
  $settings = $st->fetch() ?: ['default_currency'=>'TRY','default_currency_symbol'=>'₺','default_vat_rate'=>20];

  $ok = flash_get('ok');
  $err = flash_get('err');

  // doc_no preview (user can override)
  $docNoInvoice = next_doc_no($pdo, $cid, 'invoice');
  $docNoQuote   = next_doc_no($pdo, $cid, 'quote');

  panel_render('create', compact('customers','products','settings','docNoInvoice','docNoQuote','ok','err'));
}

/*
|--------------------------------------------------------------------------
| Documents - Create (POST)
|--------------------------------------------------------------------------
*/
function panel_create_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/create"));
  }

  $type = (string)($_POST['type'] ?? 'invoice');
  if (!in_array($type, ['invoice','quote'], true)) $type = 'invoice';

  $customerId = (int)($_POST['customer_id'] ?? 0);
  $issueDate = (string)($_POST['issue_date'] ?? date('Y-m-d'));
  $dueDate = trim((string)($_POST['due_date'] ?? ''));
  $currency = (string)($_POST['currency'] ?? 'TRY');
  $currencySymbol = (string)($_POST['currency_symbol'] ?? '₺');
  $notes = trim((string)($_POST['notes'] ?? ''));

  $shippingMode = (string)($_POST['shipping_mode'] ?? 'none'); // none|separate|line
  if (!in_array($shippingMode, ['none','separate','line'], true)) $shippingMode = 'none';
  $shippingAmount = (float)($_POST['shipping_amount'] ?? 0);
  $shippingVat = (float)($_POST['shipping_vat_rate'] ?? 20);

  $items = $_POST['items'] ?? [];
  if (!is_array($items) || count($items) === 0) {
    flash_set('err', 'En az 1 kalem eklemelisin.');
    redirect(base_path("/{$slug}/create"));
  }
  if ($customerId <= 0) {
    flash_set('err', 'Müşteri seçmelisin.');
    redirect(base_path("/{$slug}/create"));
  }

  // doc_no (user can override)
  $docNo = trim((string)($_POST['doc_no'] ?? ''));
  if ($docNo === '') {
    $docNo = next_doc_no($pdo, $cid, $type);
  } else {
    // basic validation: allow letters/numbers and -_/.
    if (!preg_match('/^[A-Za-z0-9][A-Za-z0-9\-_.\/]{1,31}$/', $docNo)) {
      flash_set('err', 'Belge No geçersiz. (Harf/rakam ve -_. / kullan)');
      redirect(base_path("/{$slug}/create"));
    }
    // ensure unique per company+type (including deleted to avoid reuse confusion)
    $st = $pdo->prepare("SELECT id FROM documents WHERE company_id=? AND type=? AND doc_no=? LIMIT 1");
    $st->execute([$cid,$type,$docNo]);
    if ($st->fetch()) {
      flash_set('err', 'Bu Belge No zaten kullanılmış.');
      redirect(base_path("/{$slug}/create"));
    }
  }

  // Totals
  $subtotal = 0.0; $vatTotal = 0.0; $grand = 0.0;
  $cleanItems = [];

  foreach ($items as $it) {
    if (!is_array($it)) continue;

    $name = trim((string)($it['name'] ?? ''));
    $desc = trim((string)($it['description'] ?? ''));
    $qty  = (float)($it['qty'] ?? 1);
    $unit = trim((string)($it['unit'] ?? 'adet')) ?: 'adet';
    $price= (float)($it['unit_price'] ?? 0);
    $vat  = (float)($it['vat_rate'] ?? 20);

    if ($name === '') continue;
    if ($qty <= 0) $qty = 1;

    $lineSub = money2($qty * $price);
    $lineVat = money2($lineSub * ($vat/100));
    $lineTot = money2($lineSub + $lineVat);

    $subtotal += $lineSub;
    $vatTotal += $lineVat;
    $grand    += $lineTot;

    $cleanItems[] = [
      'item_type' => 'product',
      'name' => $name,
      'description' => $desc ?: null,
      'qty' => $qty,
      'unit' => $unit,
      'unit_price' => $price,
      'vat_rate' => $vat,
      'line_subtotal' => $lineSub,
      'line_vat' => $lineVat,
      'line_total' => $lineTot
    ];
  }

  if (count($cleanItems) === 0) {
    flash_set('err', 'Geçerli kalem bulunamadı.');
    redirect(base_path("/{$slug}/create"));
  }

  // shipping separate
  if ($shippingMode === 'separate' && $shippingAmount > 0) {
    $shipSub = money2($shippingAmount);
    $shipVat = money2($shipSub * ($shippingVat/100));
    $shipTot = money2($shipSub + $shipVat);

    $subtotal += $shipSub;
    $vatTotal += $shipVat;
    $grand    += $shipTot;
  }

  // DB
  $pdo->beginTransaction();
  try {
    $st = $pdo->prepare("
      INSERT INTO documents
      (company_id,type,doc_no,customer_id,currency,currency_symbol,issue_date,due_date,status,notes,
       shipping_mode,shipping_amount,shipping_vat_rate,subtotal,vat_total,total)
      VALUES (?,?,?,?,?,?,?,?, 'draft', ?, ?, ?, ?, ?, ?, ?)
    ");
    $st->execute([
      $cid, $type, $docNo, $customerId, $currency, $currencySymbol,
      $issueDate,
      ($dueDate !== '' ? $dueDate : null),
      ($notes !== '' ? $notes : null),
      $shippingMode, $shippingAmount, $shippingVat,
      money2($subtotal), money2($vatTotal), money2($grand)
    ]);
    $docId = (int)$pdo->lastInsertId();

    $stItem = $pdo->prepare("
      INSERT INTO document_items
      (company_id,document_id,item_type,name,description,qty,unit,unit_price,vat_rate,line_subtotal,line_vat,line_total)
      VALUES (?,?,?,?,?,?,?,?,?,?,?,?)
    ");

    foreach ($cleanItems as $ci) {
      $stItem->execute([
        $cid,$docId,$ci['item_type'],$ci['name'],$ci['description'],$ci['qty'],$ci['unit'],
        $ci['unit_price'],$ci['vat_rate'],$ci['line_subtotal'],$ci['line_vat'],$ci['line_total']
      ]);
    }

    if ($shippingMode === 'line' && $shippingAmount > 0) {
      $shipSub = money2($shippingAmount);
      $shipVat = money2($shipSub * ($shippingVat/100));
      $shipTot = money2($shipSub + $shipVat);

      $stItem->execute([
        $cid,$docId,'shipping','Kargo',null,1,'', $shippingAmount, $shippingVat, $shipSub, $shipVat, $shipTot
      ]);
    }

    $pdo->commit();
  } catch (Throwable $e) {
    $pdo->rollBack();
    flash_set('err', 'Kayıt hatası: '.$e->getMessage());
    redirect(base_path("/{$slug}/create"));
  }

  flash_set('ok', ($type === 'invoice' ? 'Fatura' : 'Teklif') . " oluşturuldu: {$docNo}");
  redirect(base_path("/{$slug}/" . ($type === 'invoice' ? 'invoices' : 'quotes')));
}

/*
|--------------------------------------------------------------------------
| Documents - Lists
|--------------------------------------------------------------------------
*/
function panel_invoices(): void {
  panel_documents_list('invoice');
}

function panel_quotes(): void {
  panel_documents_list('quote');
}

function panel_documents_list(string $type): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $st = $pdo->prepare("
    SELECT d.id,d.doc_no,d.issue_date,d.due_date,d.total,d.currency_symbol,d.status,c.name AS customer_name
    FROM documents d
    JOIN customers c ON c.id=d.customer_id
    WHERE d.company_id=? AND d.type=? AND d.deleted_at IS NULL
    ORDER BY d.id DESC
  ");
  $st->execute([$cid,$type]);
  $rows = $st->fetchAll();

  $ok = flash_get('ok');
  $err = flash_get('err');

  panel_render($type === 'invoice' ? 'invoices' : 'quotes', compact('rows','ok','err','type'));
}

function panel_documents_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/dashboard"));
  }

  $action = (string)($_POST['action'] ?? '');
  $id = (int)($_POST['id'] ?? 0);
  $type = (string)($_POST['type'] ?? 'invoice');

  if ($id <= 0) redirect(base_path("/{$slug}/dashboard"));
  if (!in_array($type, ['invoice','quote'], true)) $type = 'invoice';

  if ($action === 'trash') {
    $st = $pdo->prepare("UPDATE documents SET deleted_at=NOW() WHERE id=? AND company_id=? AND type=?");
    $st->execute([$id,$cid,$type]);
    flash_set('ok', 'Belge çöp kutusuna taşındı.');
    redirect(base_path("/{$slug}/" . ($type==='invoice'?'invoices':'quotes')));
  }

  flash_set('err', 'Bilinmeyen işlem.');
  redirect(base_path("/{$slug}/" . ($type==='invoice'?'invoices':'quotes')));
}

/*
|--------------------------------------------------------------------------
| Trash (Documents + v0.2 Customers/Products)
|--------------------------------------------------------------------------
*/
function panel_trash(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];

  $st = $pdo->prepare("
    SELECT d.id,d.type,d.doc_no,d.deleted_at,c.name customer_name,d.total,d.currency_symbol
    FROM documents d
    JOIN customers c ON c.id=d.customer_id
    WHERE d.company_id=? AND d.deleted_at IS NOT NULL
    ORDER BY d.deleted_at DESC
  ");
  $st->execute([$cid]);
  $docs = $st->fetchAll();

  $st = $pdo->prepare("SELECT id,name,email,deleted_at,'customer' AS kind FROM customers WHERE company_id=? AND deleted_at IS NOT NULL");
  $st->execute([$cid]);
  $customers = $st->fetchAll();

  $st = $pdo->prepare("SELECT id,name,NULL AS email,deleted_at,'product' AS kind FROM products WHERE company_id=? AND deleted_at IS NOT NULL");
  $st->execute([$cid]);
  $products = $st->fetchAll();

  $ok = flash_get('ok');
  $err = flash_get('err');
  panel_render('trash', compact('docs','customers','products','ok','err'));
}

function panel_trash_restore_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/trash"));
  }

  $kind = (string)($_POST['kind'] ?? '');
  $id = (int)($_POST['id'] ?? 0);
  if ($id <= 0) redirect(base_path("/{$slug}/trash"));

  if ($kind === 'doc') {
    $st = $pdo->prepare("UPDATE documents SET deleted_at=NULL WHERE id=? AND company_id=?");
    $st->execute([$id,$cid]);
    flash_set('ok', 'Belge geri alındı.');
    redirect(base_path("/{$slug}/trash"));
  }

  if ($kind === 'customer') {
    $st = $pdo->prepare("UPDATE customers SET deleted_at=NULL WHERE id=? AND company_id=?");
    $st->execute([$id,$cid]);
    flash_set('ok', 'Müşteri geri alındı.');
  } elseif ($kind === 'product') {
    $st = $pdo->prepare("UPDATE products SET deleted_at=NULL WHERE id=? AND company_id=?");
    $st->execute([$id,$cid]);
    flash_set('ok', 'Ürün geri alındı.');
  }

  redirect(base_path("/{$slug}/trash"));
}

function panel_trash_delete_post(): void {
  $pdo = $GLOBALS['pdo'];
  $cid = (int)$GLOBALS['tenant']['id'];
  $slug = (string)$GLOBALS['tenant']['slug'];

  $csrf = (string)($_POST['csrf'] ?? '');
  if (!csrf_verify($csrf)) {
    flash_set('err', 'CSRF doğrulaması başarısız.');
    redirect(base_path("/{$slug}/trash"));
  }

  $kind = (string)($_POST['kind'] ?? '');
  $id = (int)($_POST['id'] ?? 0);
  if ($id <= 0) redirect(base_path("/{$slug}/trash"));

  if ($kind === 'doc') {
    $st = $pdo->prepare("DELETE FROM documents WHERE id=? AND company_id=?");
    $st->execute([$id,$cid]);
    flash_set('ok', 'Belge kalıcı silindi.');
    redirect(base_path("/{$slug}/trash"));
  }

  if ($kind === 'customer') {
    $st = $pdo->prepare("DELETE FROM customers WHERE id=? AND company_id=?");
    $st->execute([$id,$cid]);
    flash_set('ok', 'Müşteri kalıcı silindi.');
  } elseif ($kind === 'product') {
    $st = $pdo->prepare("DELETE FROM products WHERE id=? AND company_id=?");
    $st->execute([$id,$cid]);
    flash_set('ok', 'Ürün kalıcı silindi.');
  }

  redirect(base_path("/{$slug}/trash"));
}

/*
|--------------------------------------------------------------------------
| Helpers (en altta)
|--------------------------------------------------------------------------
*/
function money2(float $v): float { return round($v + 1e-9, 2); }

function next_doc_no(PDO $pdo, int $companyId, string $type): string {
  $st = $pdo->prepare("SELECT doc_prefix_invoice, doc_prefix_quote, doc_pad FROM company_settings WHERE company_id=? LIMIT 1");
  $st->execute([$companyId]);
  $s = $st->fetch() ?: ['doc_prefix_invoice'=>'FTR','doc_prefix_quote'=>'TKL','doc_pad'=>6];

  $prefix = ($type === 'invoice') ? (string)$s['doc_prefix_invoice'] : (string)$s['doc_prefix_quote'];
  $pad = max(3, (int)$s['doc_pad']);

  $st = $pdo->prepare("
    SELECT doc_no
    FROM documents
    WHERE company_id=? AND type=? AND deleted_at IS NULL
    ORDER BY id DESC
    LIMIT 1
  ");
  $st->execute([$companyId, $type]);
  $last = (string)($st->fetch()['doc_no'] ?? '');

  $n = 0;
  if ($last !== '' && str_starts_with($last, $prefix . '-')) {
    $num = substr($last, strlen($prefix) + 1);
    $n = (int)ltrim($num, '0');
  }
  $n++;

  return $prefix . '-' . str_pad((string)$n, $pad, '0', STR_PAD_LEFT);
}
