(() => {
  const $ = (id) => document.getElementById(id);

  const els = {
    period: $('periodSelect'),
    // KPIs
    income: $('kpiIncome'),
    incomeDelta: $('kpiIncomeDelta'),
    expense: $('kpiExpense'),
    expenseDelta: $('kpiExpenseDelta'),
    net: $('kpiNet'),
    netDelta: $('kpiNetDelta'),
    balance: $('kpiBalance'),
    cardDebt: $('kpiCardDebt'),
    portfolio: $('kpiPortfolio'),
    portfolioDelta: $('kpiPortfolioDelta'),

    // Trend
    bars: $('trendBars'),
    btn6: $('btn6'),
    btn12: $('btn12'),

    // Highlights
    worstExpenseMonth: $('worstExpenseMonth'),
    bestProfitMonth: $('bestProfitMonth'),
    topExpenseCategory: $('topExpenseCategory'),
    topIncomeCategory: $('topIncomeCategory'),
    portfolioPL: $('portfolioPL'),

    // Balance dist
    balanceDist: $('balanceDist'),

    // Investments
    invTotal: $('invTotal'),
    invDailyPL: $('invDailyPL'),
    invAlloc: $('invAlloc'),

    // Transactions
    txFilter: $('txFilter'),
    txTable: $('txTable'),

    // Goals
    goalsWidget: $('goalsWidget'),

    // Insights
    insightsList: $('insightsList'),
  };

  let state = {
    period: els.period ? els.period.value : 'this_month',
    trendWindow: 12,
    txType: 'all',
    lastPayload: null,
  };

  function fmtTRY(n) {
    const v = Number(n || 0);
    try {
      return v.toLocaleString('tr-TR', { style: 'currency', currency: 'TRY', maximumFractionDigits: 0 });
    } catch (e) {
      return '₺' + Math.round(v).toString();
    }
  }

  function fmtPct(n) {
    const v = Number(n || 0);
    const s = (v >= 0 ? '+' : '') + v.toFixed(0) + '%';
    return s;
  }

  function setDeltaBadge(el, delta) {
    if (!el) return;
    const v = Number(delta || 0);
    el.textContent = fmtPct(v);
    el.classList.remove('badge-pos','badge-neg','badge-neu');
    if (v > 0.5) el.classList.add('badge-pos');
    else if (v < -0.5) el.classList.add('badge-neg');
    else el.classList.add('badge-neu');
  }

  async function fetchDashboard() {
    const url = `api/dashboard.php?period=${encodeURIComponent(state.period)}`;
    const res = await fetch(url, { credentials: 'same-origin' });

    if (res.status === 401) {
      window.location.href = 'login.php';
      return null;
    }
    if (!res.ok) throw new Error('API error: ' + res.status);

    const j = await res.json();
    if (!j || j.ok !== true) throw new Error('Bad payload');
    return j;
  }

  function renderKpis(k) {
    if (!k) return;
    els.income && (els.income.textContent = fmtTRY(k.income));
    els.expense && (els.expense.textContent = fmtTRY(k.expense));
    els.net && (els.net.textContent = fmtTRY(k.net));
    setDeltaBadge(els.incomeDelta, k.income_delta);
    setDeltaBadge(els.expenseDelta, k.expense_delta);
    setDeltaBadge(els.netDelta, k.net_delta);

    els.balance && (els.balance.textContent = fmtTRY(k.total_balance));
    els.cardDebt && (els.cardDebt.textContent = fmtTRY(k.card_debt));
    els.portfolio && (els.portfolio.textContent = fmtTRY(k.portfolio_est));
    setDeltaBadge(els.portfolioDelta, k.portfolio_delta);
  }

  function renderTrend(trend) {
    if (!els.bars) return;
    const data = Array.isArray(trend) ? trend.slice() : [];
    // keep last N
    const w = state.trendWindow || 12;
    const slice = data.slice(Math.max(0, data.length - w));

    els.bars.innerHTML = '';
    if (!slice.length) return;

    // scale by max(abs(net), income, expense)
    let maxV = 0;
    slice.forEach(r => {
      maxV = Math.max(maxV, Math.abs(Number(r.net||0)), Math.abs(Number(r.income||0)), Math.abs(Number(r.expense||0)));
    });
    if (maxV <= 0) maxV = 1;

    slice.forEach(r => {
      const inc = Number(r.income||0);
      const exp = Number(r.expense||0);
      const net = Number(r.net||0);

      const wrap = document.createElement('div');
      wrap.className = 'col-span-1 flex flex-col items-center justify-end gap-1';

      const barWrap = document.createElement('div');
      barWrap.className = 'w-full flex items-end gap-1 h-32';

      const mk = (val, cls, title) => {
        const b = document.createElement('div');
        const h = Math.max(2, Math.round((Math.abs(val) / maxV) * 100));
        b.className = `flex-1 rounded-lg ${cls}`;
        b.style.height = `${h}%`;
        b.title = `${title}: ${fmtTRY(val)}`;
        return b;
      };

      barWrap.appendChild(mk(inc, 'bg-emerald-500/60', 'Gelir'));
      barWrap.appendChild(mk(exp, 'bg-rose-500/60', 'Gider'));
      barWrap.appendChild(mk(net, 'bg-slate-500/50', 'Net'));

      const lab = document.createElement('div');
      lab.className = 'text-[10px] muted truncate w-full text-center';
      // bucket is YYYY-MM-DD or YYYY-MM-01
      const bucket = String(r.bucket||'');
      lab.textContent = bucket.slice(5,7) + '/' + bucket.slice(2,4);

      wrap.appendChild(barWrap);
      wrap.appendChild(lab);
      els.bars.appendChild(wrap);
    });
  }

  function renderBalanceDist(accounts) {
    if (!els.balanceDist) return;
    const arr = Array.isArray(accounts) ? accounts : [];
    els.balanceDist.innerHTML = '';
    if (!arr.length) return;

    const maxAbs = Math.max(...arr.map(a => Math.abs(Number(a.balance||0))), 1);

    arr.forEach(a => {
      const row = document.createElement('div');
      row.className = 'mini-box';

      const top = document.createElement('div');
      top.className = 'flex items-center justify-between gap-2';
      top.innerHTML = `<span class="text-sm font-medium">${escapeHtml(String(a.name||''))}</span><span class="text-sm font-semibold">${fmtTRY(a.balance)}</span>`;

      const bar = document.createElement('div');
      bar.className = 'mt-2 h-2 rounded-full bg-slate-200 dark:bg-slate-800 overflow-hidden';

      const fill = document.createElement('div');
      fill.className = 'h-full bg-slate-900/70 dark:bg-white/70';
      fill.style.width = `${Math.min(100, (Math.abs(Number(a.balance||0))/maxAbs)*100)}%`;

      bar.appendChild(fill);
      row.appendChild(top);
      row.appendChild(bar);

      els.balanceDist.appendChild(row);
    });
  }

  function renderHighlights(h) {
    const dash = (el) => { if (el) el.textContent = '—'; };

    if (!h) {
      dash(els.worstExpenseMonth); dash(els.bestProfitMonth);
      dash(els.topExpenseCategory); dash(els.topIncomeCategory); dash(els.portfolioPL);
      return;
    }

    if (els.worstExpenseMonth) {
      els.worstExpenseMonth.textContent = h.max_expense_bucket
        ? `${(h.max_expense_bucket.bucket || '').slice(0,7)} • ${fmtTRY(h.max_expense_bucket.value)}`
        : '—';
    }
    if (els.bestProfitMonth) {
      els.bestProfitMonth.textContent = h.best_net_bucket
        ? `${(h.best_net_bucket.bucket || '').slice(0,7)} • ${fmtTRY(h.best_net_bucket.value)}`
        : '—';
    }
    if (els.topExpenseCategory) {
      els.topExpenseCategory.textContent = h.top_expense_category
        ? `${h.top_expense_category.name} • ${fmtTRY(h.top_expense_category.total)}`
        : '—';
    }
    if (els.topIncomeCategory) {
      els.topIncomeCategory.textContent = h.top_income_category
        ? `${h.top_income_category.name} • ${fmtTRY(h.top_income_category.total)}`
        : '—';
    }
    if (els.portfolioPL) {
      els.portfolioPL.textContent = (h.portfolio_pl === null || typeof h.portfolio_pl === 'undefined')
        ? '—'
        : fmtTRY(h.portfolio_pl);
    }
  }

  function renderInvestments(inv) {
    if (!inv) return;
    els.invTotal && (els.invTotal.textContent = fmtTRY(inv.total_portfolio));
    els.invDailyPL && (els.invDailyPL.textContent = fmtTRY(inv.daily_pl));

    if (els.invAlloc) {
      const arr = Array.isArray(inv.by_account) ? inv.by_account : [];
      els.invAlloc.innerHTML = '';
      if (!arr.length) {
        els.invAlloc.innerHTML = `<div class="text-sm muted">Henüz yatırım hesabı yok.</div>`;
        return;
      }

      const total = arr.reduce((s,a)=>s+Number(a.value||0),0) || 1;
      arr.forEach(a => {
        const pct = (Number(a.value||0) / total) * 100;
        const row = document.createElement('div');
        row.className = 'mini-box';
        row.innerHTML = `
          <div class="flex items-center justify-between gap-2">
            <span class="text-sm font-medium">${escapeHtml(String(a.name||''))}</span>
            <span class="text-sm font-semibold">${fmtTRY(a.value)}</span>
          </div>
          <div class="mt-2 h-2 rounded-full bg-slate-200 dark:bg-slate-800 overflow-hidden">
            <div class="h-full bg-indigo-500/70" style="width:${Math.min(100,pct)}%"></div>
          </div>
        `;
        els.invAlloc.appendChild(row);
      });
    }
  }

  function renderTxTable(lastTx) {
    if (!els.txTable) return;
    const arr = Array.isArray(lastTx) ? lastTx : [];
    const filtered = state.txType === 'all' ? arr : arr.filter(x => x.type === state.txType);

    els.txTable.innerHTML = '';
    filtered.forEach(x => {
      const tr = document.createElement('tr');
      tr.className = 'hover:bg-slate-50 dark:hover:bg-slate-900/30';
      tr.innerHTML = `
        <td class="py-2">${escapeHtml(String(x.date||'').slice(0,10))}</td>
        <td class="py-2">${typeLabel(x.type)}</td>
        <td class="py-2">${escapeHtml(String(x.desc||''))}</td>
        <td class="py-2 text-right font-semibold">${fmtTRY(x.amount)}</td>
      `;
      els.txTable.appendChild(tr);
    });
  }

  function renderGoals(goals) {
    if (!els.goalsWidget) return;
    const arr = Array.isArray(goals) ? goals : [];
    els.goalsWidget.innerHTML = '';

    if (!arr.length) {
      els.goalsWidget.innerHTML = `<div class="text-sm muted md:col-span-3">Henüz hedef yok. <a class="underline" href="goals.php">Hedef ekle</a></div>`;
      return;
    }

    arr.forEach(g => {
      const card = document.createElement('div');
      card.className = 'mini-box';

      const pct = Math.max(0, Math.min(100, Number(g.pct||0)));
      const date = g.target_date ? String(g.target_date).slice(0,10) : 'Tarih yok';

      card.innerHTML = `
        <div class="flex items-start justify-between gap-2">
          <div>
            <p class="font-semibold">${escapeHtml(String(g.name||''))}</p>
            <p class="text-xs muted mt-0.5">${escapeHtml(date)}</p>
          </div>
          <span class="badge badge-neu">${pct.toFixed(0)}%</span>
        </div>
        <div class="mt-3 h-2 rounded-full bg-slate-200 dark:bg-slate-800 overflow-hidden">
          <div class="h-full bg-emerald-500/70" style="width:${pct}%"></div>
        </div>
        <div class="mt-3 flex items-center justify-between text-xs">
          <span class="muted">Birikim</span>
          <span class="font-semibold">${fmtTRY(g.contributed)} / ${fmtTRY(g.target_amount)}</span>
        </div>
      `;
      els.goalsWidget.appendChild(card);
    });
  }

  function renderInsights(list) {
    if (!els.insightsList) return;
    const arr = Array.isArray(list) ? list : [];
    els.insightsList.innerHTML = '';
    if (!arr.length) {
      els.insightsList.innerHTML = `<li class="text-sm muted">Henüz içgörü yok.</li>`;
      return;
    }
    arr.forEach(t => {
      const li = document.createElement('li');
      li.className = 'mini-box';
      li.textContent = String(t);
      els.insightsList.appendChild(li);
    });
  }

  function typeLabel(t) {
    if (t === 'income') return 'Gelir';
    if (t === 'expense') return 'Gider';
    if (t === 'transfer') return 'Transfer';
    return escapeHtml(String(t||''));
  }

  function escapeHtml(s) {
    return s
      .replaceAll('&','&amp;')
      .replaceAll('<','&lt;')
      .replaceAll('>','&gt;')
      .replaceAll('"','&quot;')
      .replaceAll("'","&#039;");
  }

  async function load() {
    try {
      const data = await fetchDashboard();
      if (!data) return;
      state.lastPayload = data;

      renderKpis(data.kpis);
      renderTrend(data.trend);
      renderBalanceDist(data.accounts);
      renderHighlights(data.highlights);
      renderInvestments(data.investments);
      renderTxTable(data.last_tx);
      renderGoals(data.goals);
      renderInsights(data.insights);
    } catch (e) {
      // keep UI defaults; optionally log
      console.error(e);
    }
  }

  // events
  if (els.period) {
    els.period.addEventListener('change', () => {
      state.period = els.period.value;
      load();
    });
  }

  if (els.btn6) els.btn6.addEventListener('click', () => { state.trendWindow = 6; load(); });
  if (els.btn12) els.btn12.addEventListener('click', () => { state.trendWindow = 12; load(); });

  if (els.txFilter) {
    els.txFilter.addEventListener('change', () => {
      state.txType = els.txFilter.value;
      if (state.lastPayload) renderTxTable(state.lastPayload.last_tx);
    });
  }

  // initial
  load();
})();
