// reports.js
(function(){
  const $ = (id)=>document.getElementById(id);

  function fmtTRY(n){
    n = Number(n||0);
    const s = Math.round(n).toString();
    // basic thousands separator for TR
    return "₺" + s.replace(/\B(?=(\d{3})+(?!\d))/g, ".");
  }

  function qs(){
    const period = $('periodSelect').value;
    const year = Number($('yearSelect').value||0);
    const month = Number($('monthSelect').value||0);
    const p = new URLSearchParams();
    if (period) p.set('period', period);
    if (year) p.set('year', String(year));
    if (month) p.set('month', String(month));
    return p;
  }

  function apply(){
    const p = qs();
    const url = new URL(window.location.href);
    url.search = p.toString();
    window.location.href = url.toString();
  }

  function setTab(tabId){
    document.querySelectorAll('.tab-btn').forEach(b=>{
      b.classList.toggle('tab-active', b.dataset.tab === tabId);
    });
    document.querySelectorAll('.tab-pane').forEach(p=>{
      p.classList.toggle('hidden', p.id !== tabId);
    });
  }

  function rowHTML(cols){
    return "<tr>" + cols.map(c=>`<td class="py-2 ${c.cls||''}">${c.html}</td>`).join('') + "</tr>";
  }

  function fillTable(tbodyId, rowsHtml, emptyText){
    const tb = $(tbodyId);
    if (!tb) return;
    tb.innerHTML = rowsHtml.length ? rowsHtml.join('') : `<tr><td colspan="10" class="py-6 text-center muted">${emptyText||'Kayıt yok.'}</td></tr>`;
  }

  async function load(){
    const p = qs();
    const apiUrl = "api/reports.php?" + p.toString();
    const res = await fetch(apiUrl, {headers: {"Accept":"application/json"}});
    const data = await res.json();

    if (!data || !data.ok){
      $('kpiIncome').textContent = "₺0";
      $('kpiExpense').textContent = "₺0";
      $('kpiNet').textContent = "₺0";
      return;
    }

    $('kpiIncome').textContent = fmtTRY(data.kpis?.income);
    $('kpiExpense').textContent = fmtTRY(data.kpis?.expense);
    $('kpiNet').textContent = fmtTRY(data.kpis?.net);

    // print header
    const start = data.range?.start || '';
    const end = data.range?.end || '';
    const pr = document.getElementById('printRange');
    if (pr) pr.textContent = (start && end) ? `${start} → ${end}` : '—';
    const pd = document.getElementById('printDate');
    if (pd) pd.textContent = new Date().toISOString().slice(0,10);

    // summary
    $('avgDailyExpense').textContent = fmtTRY(data.summary?.avg_daily_expense || 0);
    $('maxExpenseDay').textContent = data.summary?.max_expense_day || '—';
    $('maxIncomeDay').textContent = data.summary?.max_income_day || '—';

    // last tx
    const txRows = (data.last_tx||[]).map(r=>{
      const type = r.type || '';
      const badge = type==='income'?'badge-pos':(type==='expense'?'badge-neg':'badge-neu');
      const label = type==='income'?'Gelir':(type==='expense'?'Gider':'Transfer');
      const amtSign = type==='expense' ? '-' : '+';
      return rowHTML([
        {html: `<span class="muted">${r.date||''}</span>`},
        {html: `<span class="badge ${badge}">${label}</span>`},
        {html: `<span class="font-medium">${(r.desc||'')}</span>`},
        {html: `<span class="font-semibold ${type==='expense'?'text-rose-600 dark:text-rose-400':'text-emerald-600 dark:text-emerald-400'}">${amtSign}${fmtTRY(r.amount||0)}</span>`, cls:'text-right'}
      ]);
    });
    fillTable('lastTxTable', txRows, 'İşlem yok.');

    // insights
    const ins = document.getElementById('insights');
    if (ins){
      ins.innerHTML = '';
      (data.insights||[]).forEach(t=>{
        const li = document.createElement('li');
        li.className = 'p-2 rounded-xl bg-slate-50 dark:bg-slate-900/40 border border-slate-200 dark:border-slate-800';
        li.textContent = t;
        ins.appendChild(li);
      });
      if (!(data.insights||[]).length){
        const li = document.createElement('li');
        li.className = 'muted';
        li.textContent = 'Şimdilik içgörü yok.';
        ins.appendChild(li);
      }
    }

    // category tables
    const expRows = (data.expense_by_cat||[]).map(r=>{
      return rowHTML([
        {html: `<span class="font-medium">${r.name||'-'}</span>`},
        {html: `<span class="font-semibold text-right block">${fmtTRY(r.amount||0)}</span>`, cls:'text-right'},
        {html: `<span class="muted text-right block">${Number(r.pct||0).toFixed(1)}%</span>`, cls:'text-right'},
      ]);
    });
    fillTable('expenseCatTable', expRows, 'Gider yok.');

    const incRows = (data.income_by_cat||[]).map(r=>{
      return rowHTML([
        {html: `<span class="font-medium">${r.name||'-'}</span>`},
        {html: `<span class="font-semibold text-right block">${fmtTRY(r.amount||0)}</span>`, cls:'text-right'},
        {html: `<span class="muted text-right block">${Number(r.pct||0).toFixed(1)}%</span>`, cls:'text-right'},
      ]);
    });
    fillTable('incomeCatTable', incRows, 'Gelir yok.');

    // account table
    const accRows = (data.by_account||[]).map(r=>{
      return rowHTML([
        {html: `<span class="font-semibold">${r.name||'-'}</span>`},
        {html: `<span class="text-right block">${fmtTRY(r.income||0)}</span>`, cls:'text-right'},
        {html: `<span class="text-right block">${fmtTRY(r.expense||0)}</span>`, cls:'text-right'},
        {html: `<span class="text-right block">${fmtTRY(r.transfer_net||0)}</span>`, cls:'text-right'},
        {html: `<span class="text-right block font-bold">${fmtTRY(r.net||0)}</span>`, cls:'text-right'},
      ]);
    });
    fillTable('accountTable', accRows, 'Hesap verisi yok.');

    // cards
    $('cardDebt').textContent = fmtTRY(data.cards?.total_debt || 0);
    $('cardSpend').textContent = fmtTRY(data.cards?.spend || 0);
    $('cardPayments').textContent = fmtTRY(data.cards?.payments || 0);
    $('cardCashAdvance').textContent = fmtTRY(data.cards?.cash_advance || 0);

    // investments
    $('invPortfolio').textContent = fmtTRY(data.investments?.portfolio_est || 0);
    $('invPL').textContent = fmtTRY(data.investments?.period_pl || 0);
    $('invAssetCount').textContent = String(data.investments?.asset_count || 0);

    const invRows = (data.investments?.by_asset||[]).map(r=>{
      return rowHTML([
        {html: `<span class="font-medium">${r.symbol||'-'}</span>`},
        {html: `<span class="text-right block">${r.qty||0}</span>`, cls:'text-right'},
        {html: `<span class="text-right block">${fmtTRY(r.est_try||0)}</span>`, cls:'text-right'},
      ]);
    });
    fillTable('invAssetTable', invRows, 'Varlık yok.');
  }

  // Events
  document.addEventListener('click', (e)=>{
    const b = e.target.closest('.tab-btn');
    if (b && b.dataset.tab) setTab(b.dataset.tab);
  });

  $('applyBtn')?.addEventListener('click', apply);
  $('pdfBtn')?.addEventListener('click', ()=> window.print());

  // default tab
  setTab('tab-summary');

  // load
  load().catch(()=>{});
})();
