<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/helpers.php';

$uid = current_user_id();
if (!$uid) json_out(['ok'=>false,'error'=>'unauthorized'], 401);

$period = (string)($_GET['period'] ?? 'this_month');
[$start, $end] = period_range($period);

// ---------- helpers ----------
function sum_income_expense(PDO $pdo, int $uid, string $start, string $end): array {
  $st = $pdo->prepare("
    SELECT
      COALESCE(SUM(CASE WHEN type='income'  THEN amount ELSE 0 END),0) AS income,
      COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
      AND occurred_at BETWEEN ? AND ?
      AND type IN ('income','expense')
  ");
  $st->execute([$uid, $start, $end]);
  $row = $st->fetch() ?: ['income'=>0,'expense'=>0];
  $income = (float)($row['income'] ?? 0);
  $expense = (float)($row['expense'] ?? 0);
  return ['income'=>$income, 'expense'=>$expense, 'net'=>$income-$expense];
}

function prev_period_range(string $period): ?array {
  // returns [start,end] for comparison period
  $today = new DateTime('today');
  if ($period === 'this_month') {
    $s = (new DateTime('first day of last month'))->format('Y-m-d');
    $e = (new DateTime('last day of last month'))->format('Y-m-d');
    return [$s,$e];
  }
  if ($period === 'last_month') {
    // compare to the month before last
    $s = (new DateTime('first day of -2 month'))->format('Y-m-d');
    $e = (new DateTime('last day of -2 month'))->format('Y-m-d');
    return [$s,$e];
  }
  if ($period === 'this_year') {
    $y = (int)$today->format('Y') - 1;
    return [sprintf('%04d-01-01',$y), sprintf('%04d-12-31',$y)];
  }
  return null;
}

function pct_delta(float $now, float $prev): float {
  if ($prev == 0.0) return $now == 0.0 ? 0.0 : 100.0;
  return (($now - $prev) / abs($prev)) * 100.0;
}

// ---------- KPIs ----------
$sum = sum_income_expense($pdo, $uid, $start, $end);
$income  = $sum['income'];
$expense = $sum['expense'];
$net     = $sum['net'];

$prev = prev_period_range($period);
$incomeDelta = 0.0; $expenseDelta = 0.0; $netDelta = 0.0;
if ($prev) {
  [$ps,$pe] = $prev;
  $prevSum = sum_income_expense($pdo, $uid, $ps, $pe);
  $incomeDelta  = pct_delta($income,  (float)$prevSum['income']);
  $expenseDelta = pct_delta($expense, (float)$prevSum['expense']);
  $netDelta     = pct_delta($net,     (float)$prevSum['net']);
}

// ---------- Accounts balances ----------
$acc = $pdo->prepare("
  SELECT a.id, a.name,
    (a.opening_balance
      + COALESCE((
        SELECT SUM(
          CASE
            WHEN t.type='income'  AND t.account_id=a.id THEN t.amount
            WHEN t.type='expense' AND t.account_id=a.id THEN -t.amount
            WHEN t.type='transfer' AND t.to_account_id=a.id THEN t.amount
            WHEN t.type='transfer' AND t.from_account_id=a.id THEN -t.amount
            ELSE 0
          END
        )
        FROM transactions t
        WHERE t.user_id=a.user_id AND t.deleted_at IS NULL
      ),0)
    ) AS balance
  FROM accounts a
  WHERE a.user_id=? AND a.deleted_at IS NULL AND a.is_active=1
  ORDER BY a.name
");
$acc->execute([$uid]);
$accounts = $acc->fetchAll() ?: [];

$total_balance = 0.0;
$accounts_out = [];
foreach ($accounts as $a) {
  $bal = (float)$a['balance'];
  $total_balance += $bal;
  $accounts_out[] = [
    'id' => (int)$a['id'],
    'name' => (string)$a['name'],
    'balance' => $bal,
  ];
}

// ---------- Credit card debt (best-effort) ----------
$cardDebt = 0.0;
try {
  // common schema: credit_cards.used_amount
  $st = $pdo->prepare("SELECT COALESCE(SUM(used_amount),0) AS debt FROM credit_cards WHERE user_id=? AND deleted_at IS NULL AND is_active=1");
  $st->execute([$uid]);
  $row = $st->fetch();
  $cardDebt = (float)($row['debt'] ?? 0);
} catch (Throwable $e) {
  $cardDebt = 0.0;
}

// ---------- Investments (placeholder until module wired) ----------
$portfolio_est = 0.0;
try {
  // If you later store investment account TRY value, compute it here.
  $portfolio_est = 0.0;
} catch (Throwable $e) {
  $portfolio_est = 0.0;
}

// ---------- Last transactions ----------
$st = $pdo->prepare("
  SELECT id, occurred_at, type, amount, description
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
  ORDER BY occurred_at DESC, id DESC
  LIMIT 12
");
$st->execute([$uid]);
$last = $st->fetchAll() ?: [];
$last_out = [];
foreach ($last as $r) {
  $last_out[] = [
    'id' => (int)$r['id'],
    'date' => (string)$r['occurred_at'],
    'type' => (string)$r['type'],
    'amount' => (float)$r['amount'],
    'desc' => (string)($r['description'] ?? ''),
  ];
}

// ---------- Trend (last 24 months) ----------
$trend = [];
try {
  $st = $pdo->prepare("
    SELECT DATE_FORMAT(occurred_at,'%Y-%m') AS ym,
      COALESCE(SUM(CASE WHEN type='income'  THEN amount ELSE 0 END),0) AS income,
      COALESCE(SUM(CASE WHEN type='expense' THEN amount ELSE 0 END),0) AS expense
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
      AND occurred_at >= DATE_SUB(CURDATE(), INTERVAL 24 MONTH)
      AND type IN ('income','expense')
    GROUP BY ym
    ORDER BY ym ASC
  ");
  $st->execute([$uid]);
  $rows = $st->fetchAll() ?: [];

  // create continuous months series
  $map = [];
  foreach ($rows as $r) {
    $map[(string)$r['ym']] = [
      'income' => (float)$r['income'],
      'expense' => (float)$r['expense'],
    ];
  }
  $d = new DateTime('first day of this month');
  $d->modify('-23 months');
  for ($i=0; $i<24; $i++) {
    $ym = $d->format('Y-m');
    $inc = $map[$ym]['income'] ?? 0.0;
    $exp = $map[$ym]['expense'] ?? 0.0;
    $trend[] = ['m'=>$ym, 'income'=>$inc, 'expense'=>$exp, 'net'=>$inc-$exp];
    $d->modify('+1 month');
  }
} catch (Throwable $e) {
  $trend = [];
}

// ---------- Highlights (best-effort from last 12 points of trend + categories in period) ----------
$highlights = [
  'worstExpenseMonth' => '',
  'bestProfitMonth' => '',
  'topExpenseCategory' => '',
  'topIncomeCategory' => '',
];

if ($trend) {
  $last12 = array_slice($trend, -12);
  $worst = null; $best = null;
  foreach ($last12 as $t) {
    if ($worst === null || $t['expense'] > $worst['expense']) $worst = $t;
    if ($best === null  || $t['net']    > $best['net'])       $best  = $t;
  }
  if ($worst) $highlights['worstExpenseMonth'] = $worst['m'].' • '.fmt_try((float)$worst['expense']).' gider';
  if ($best)  $highlights['bestProfitMonth']   = $best['m'].' • '.fmt_try((float)$best['net']).' net';
}

// top categories in selected period
try {
  // Expense categories
  $st = $pdo->prepare("
    SELECT c.name, SUM(t.amount) AS total
    FROM transactions t
    JOIN categories c ON c.id=t.category_id
    WHERE t.user_id=? AND t.deleted_at IS NULL
      AND t.type='expense'
      AND t.occurred_at BETWEEN ? AND ?
    GROUP BY c.id, c.name
    ORDER BY total DESC
    LIMIT 1
  ");
  $st->execute([$uid, $start, $end]);
  $top = $st->fetch();
  if ($top && (float)$top['total'] > 0) {
    $st2 = $pdo->prepare("
      SELECT COALESCE(SUM(amount),0) AS total
      FROM transactions
      WHERE user_id=? AND deleted_at IS NULL
        AND type='expense' AND occurred_at BETWEEN ? AND ?
    ");
    $st2->execute([$uid, $start, $end]);
    $tot = (float)(($st2->fetch()['total'] ?? 0));
    $pct = $tot>0 ? round(((float)$top['total']/$tot)*100, 1) : 0.0;
    $highlights['topExpenseCategory'] = (string)$top['name'].' • %'.$pct;
  }
} catch (Throwable $e) {}

try {
  // Income categories
  $st = $pdo->prepare("
    SELECT c.name, SUM(t.amount) AS total
    FROM transactions t
    JOIN categories c ON c.id=t.category_id
    WHERE t.user_id=? AND t.deleted_at IS NULL
      AND t.type='income'
      AND t.occurred_at BETWEEN ? AND ?
    GROUP BY c.id, c.name
    ORDER BY total DESC
    LIMIT 1
  ");
  $st->execute([$uid, $start, $end]);
  $top = $st->fetch();
  if ($top && (float)$top['total'] > 0) {
    $st2 = $pdo->prepare("
      SELECT COALESCE(SUM(amount),0) AS total
      FROM transactions
      WHERE user_id=? AND deleted_at IS NULL
        AND type='income' AND occurred_at BETWEEN ? AND ?
    ");
    $st2->execute([$uid, $start, $end]);
    $tot = (float)(($st2->fetch()['total'] ?? 0));
    $pct = $tot>0 ? round(((float)$top['total']/$tot)*100, 1) : 0.0;
    $highlights['topIncomeCategory'] = (string)$top['name'].' • %'.$pct;
  }
} catch (Throwable $e) {}

// ---------- Output ----------
json_out([
  'ok' => true,
  'period' => $period,
  'range' => ['start'=>$start,'end'=>$end],
  'kpis' => [
    'income' => $income,
    'income_delta' => round($incomeDelta, 1),
    'expense' => $expense,
    'expense_delta' => round($expenseDelta, 1),
    'net' => $net,
    'net_delta' => round($netDelta, 1),
    'total_balance' => $total_balance,
    'card_debt' => $cardDebt,
    'portfolio_est' => $portfolio_est,
  ],
  'accounts' => $accounts_out,
  'last_tx' => $last_out,
  'trend' => $trend,
  'highlights' => $highlights,
]);
