<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';
if (file_exists(__DIR__ . '/../inc/csrf.php')) require __DIR__ . '/../inc/csrf.php';

$uid = function_exists('require_user_id') ? require_user_id($pdo) : (current_user_id() ?? 0);
if ($uid <= 0) {
  flash_set('error', 'Users tablosunda kullanıcı bulunamadı. Lütfen 1 kullanıcı ekle.');
  header('Location: ' . page_url('index.php'));
  exit;
}

function post_str(string $k): string { return trim((string)($_POST[$k] ?? '')); }
function post_int(string $k): int { return (int)($_POST[$k] ?? 0); }
function post_float(string $k): float { return (float)($_POST[$k] ?? 0); }

function csrf_ok(): bool {
  if (!function_exists('csrf_verify')) return true;
  return csrf_verify((string)($_POST['csrf'] ?? ''));
}
function csrf_field(): string {
  if (!function_exists('csrf_token')) return '';
  return '<input type="hidden" name="csrf" value="'.h(csrf_token()).'">';
}

function ensure_goal_expense_category(PDO $pdo, int $uid): ?int {
  // “Hedef Birikimi” adlı expense kategorisi yoksa oluşturur.
  $st = $pdo->prepare("SELECT id FROM categories WHERE user_id=? AND type='expense' AND name=? AND deleted_at IS NULL LIMIT 1");
  $st->execute([$uid, 'Hedef Birikimi']);
  $id = $st->fetchColumn();
  if ($id) return (int)$id;

  try {
    $ins = $pdo->prepare("INSERT INTO categories (user_id,type,name,color,sort_order,is_active,created_at) VALUES (?,?,?,?,?,1,NOW())");
    $ins->execute([$uid, 'expense', 'Hedef Birikimi', null, 0]);
    return (int)$pdo->lastInsertId();
  } catch (Throwable) {
    // Kategori eklenemese bile katkı kaydı çalışsın (category_id null kalır)
    return null;
  }
}

function insert_goal_expense_tx(PDO $pdo, int $uid, string $date, float $amount, string $currency, ?int $categoryId,
                               int $accountId, string $desc): void {
  $st = $pdo->prepare("
    INSERT INTO transactions
      (user_id, kind, occurred_on, amount, currency, category_id, description, source_type, source_id, dest_type, dest_id)
    VALUES
      (?, 'expense', ?, ?, ?, ?, ?, 'account', ?, 'none', NULL)
  ");
  $st->execute([$uid, $date, $amount, $currency, $categoryId, $desc, $accountId]);
}

/* -------------------------------------------------------
   Actions
------------------------------------------------------- */
$action = (string)($_POST['action'] ?? '');
if ($action !== '') {
  try {
    if (!csrf_ok()) throw new RuntimeException('Oturum doğrulaması (CSRF) başarısız.');

    if ($action === 'add_goal') {
      $name = post_str('name');
      $targetAmount = post_float('target_amount');
      $targetDate = post_str('target_date');

      if ($name === '') throw new RuntimeException('Hedef adı zorunlu.');
      if ($targetAmount <= 0) throw new RuntimeException('Hedef tutarı 0’dan büyük olmalı.');
      if ($targetDate === '') throw new RuntimeException('Bitiş tarihi zorunlu.');

      $st = $pdo->prepare("INSERT INTO goals (user_id,name,target_amount,target_date,status,created_at) VALUES (?,?,?,?, 'active', NOW())");
      $st->execute([$uid, $name, $targetAmount, $targetDate]);

      flash_set('success', 'Hedef eklendi ✅');
      header('Location: ' . page_url('goals.php'));
      exit;
    }

    if ($action === 'update_goal') {
      $id = post_int('id');
      $name = post_str('name');
      $targetAmount = post_float('target_amount');
      $targetDate = post_str('target_date');
      $status = post_str('status');

      if ($id <= 0) throw new RuntimeException('Geçersiz hedef.');
      if ($name === '') throw new RuntimeException('Hedef adı zorunlu.');
      if ($targetAmount <= 0) throw new RuntimeException('Hedef tutarı 0’dan büyük olmalı.');
      if ($targetDate === '') throw new RuntimeException('Bitiş tarihi zorunlu.');
      if (!in_array($status, ['active','paused','completed','canceled'], true)) $status = 'active';

      $st = $pdo->prepare("UPDATE goals SET name=?, target_amount=?, target_date=?, status=?, updated_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
      $st->execute([$name, $targetAmount, $targetDate, $status, $id, $uid]);

      flash_set('success', 'Hedef güncellendi ✅');
      header('Location: ' . page_url('goals.php'));
      exit;
    }

    if ($action === 'delete_goal') {
      $id = post_int('id');
      if ($id <= 0) throw new RuntimeException('Geçersiz hedef.');

      $pdo->beginTransaction();
      $st = $pdo->prepare("UPDATE goals SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
      $st->execute([$id, $uid]);

      // katkıları da soft-delete (opsiyonel ama temiz)
      $st = $pdo->prepare("UPDATE goal_contributions SET deleted_at=NOW() WHERE goal_id=? AND user_id=? AND deleted_at IS NULL");
      $st->execute([$id, $uid]);

      $pdo->commit();

      flash_set('success', 'Hedef çöp kutusuna taşındı 🗑️');
      header('Location: ' . page_url('goals.php'));
      exit;
    }

    if ($action === 'add_contribution') {
      $goalId = post_int('goal_id');
      $accountId = post_int('account_id');
      $amount = post_float('amount');
      $date = post_str('contributed_at');
      $note = post_str('note');

      if ($goalId <= 0) throw new RuntimeException('Geçersiz hedef.');
      if ($accountId <= 0) throw new RuntimeException('Katkı için hesap seçmelisin.');
      if ($amount <= 0) throw new RuntimeException('Katkı tutarı 0’dan büyük olmalı.');
      if ($date === '') $date = date('Y-m-d');

      // goal belongs to user?
      $st = $pdo->prepare("SELECT name,target_amount,status FROM goals WHERE id=? AND user_id=? AND deleted_at IS NULL LIMIT 1");
      $st->execute([$goalId, $uid]);
      $g = $st->fetch();
      if (!$g) throw new RuntimeException('Hedef bulunamadı.');
      if (in_array((string)$g['status'], ['completed','canceled'], true)) {
        throw new RuntimeException('Bu hedef kapalı. Önce durumu aktif yap.');
      }

      // account belongs to user?
      $st = $pdo->prepare("SELECT id,currency,name FROM accounts WHERE id=? AND user_id=? AND deleted_at IS NULL AND is_active=1 LIMIT 1");
      $st->execute([$accountId, $uid]);
      $acc = $st->fetch();
      if (!$acc) throw new RuntimeException('Hesap bulunamadı.');

      $currency = strtoupper((string)($acc['currency'] ?? 'TRY'));
      if (!in_array($currency, ['TRY','USD','EUR'], true)) $currency = 'TRY';

      $pdo->beginTransaction();

      // 1) goal_contributions
      $ins = $pdo->prepare("
        INSERT INTO goal_contributions (user_id, goal_id, account_id, amount, contributed_at, note, created_at)
        VALUES (?,?,?,?,?,?,NOW())
      ");
      $ins->execute([$uid, $goalId, $accountId, $amount, $date, ($note !== '' ? $note : null)]);

      // 2) transactions as expense (goal saving is an expense)
      $catId = ensure_goal_expense_category($pdo, $uid);
      $desc = 'Hedef Birikimi • ' . (string)$g['name'];
      if ($note !== '') $desc .= ' — ' . $note;

      insert_goal_expense_tx($pdo, $uid, $date, $amount, $currency, $catId, $accountId, $desc);

      $pdo->commit();

      flash_set('success', 'Hedef katkısı eklendi ✅ (gider olarak işlendi)');
      header('Location: ' . page_url('goals.php') . '#goal-' . $goalId);
      exit;
    }

    if ($action === 'delete_contribution') {
      $id = post_int('id');
      $goalId = post_int('goal_id');
      if ($id <= 0) throw new RuntimeException('Geçersiz katkı.');

      // Not: Bu katkıya bağlı “transactions” kaydını ilişkilendirmek için tabloda FK yok.
      // Şimdilik sadece goal_contributions soft-delete yapıyoruz.
      $st = $pdo->prepare("UPDATE goal_contributions SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
      $st->execute([$id, $uid]);

      flash_set('success', 'Katkı çöp kutusuna taşındı 🗑️');
      header('Location: ' . page_url('goals.php') . '#goal-' . $goalId);
      exit;
    }

    throw new RuntimeException('Geçersiz işlem.');
  } catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    flash_set('error', $e->getMessage());
    header('Location: ' . page_url('goals.php'));
    exit;
  }
}

/* -------------------------------------------------------
   Data
------------------------------------------------------- */
$accSt = $pdo->prepare("SELECT id,name,currency FROM accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$accSt->execute([$uid]);
$accounts = $accSt->fetchAll();

$goalsSt = $pdo->prepare("
  SELECT g.*,
         COALESCE(SUM(CASE WHEN gc.deleted_at IS NULL THEN gc.amount ELSE 0 END),0) AS saved_amount
  FROM goals g
  LEFT JOIN goal_contributions gc
    ON gc.goal_id = g.id AND gc.user_id = g.user_id
  WHERE g.user_id=? AND g.deleted_at IS NULL
  GROUP BY g.id
  ORDER BY (g.status='active') DESC, g.target_date ASC, g.id DESC
");
$goalsSt->execute([$uid]);
$goals = $goalsSt->fetchAll();

// Contributions for each goal (last 5)
$contribSt = $pdo->prepare("
  SELECT gc.*, a.name AS account_name
  FROM goal_contributions gc
  LEFT JOIN accounts a ON a.id=gc.account_id
  WHERE gc.user_id=? AND gc.goal_id=? AND gc.deleted_at IS NULL
  ORDER BY gc.contributed_at DESC, gc.id DESC
  LIMIT 5
");

require __DIR__ . '/../inc/header.php';
?>

<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
  <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
    <div>
      <h2 class="text-lg font-semibold">🎯 Hedefler</h2>
      <p class="text-sm opacity-70">Hedef ekle, katkı yap; katkılar otomatik gider olarak işlenir.</p>
    </div>
    <a href="<?= h(page_url('trash.php')) ?>?entity=goals"
       class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
      🗑️ Çöp Kutusu
    </a>
  </div>
</div>

<div class="mt-4 grid grid-cols-1 xl:grid-cols-3 gap-4">
  <!-- Add goal -->
  <div class="xl:col-span-1 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h3 class="text-lg font-semibold">➕ Yeni Hedef Ekle</h3>

    <form class="mt-4 space-y-3" method="post">
      <?= csrf_field() ?>
      <input type="hidden" name="action" value="add_goal">

      <div>
        <label class="text-sm opacity-70">Hedef Adı</label>
        <input name="name" placeholder="Örn: Tatil 2026"
               class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
      </div>

      <div>
        <label class="text-sm opacity-70">Hedef Tutar (₺)</label>
        <input name="target_amount" type="number" step="0.01" min="0" required
               class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
      </div>

      <div>
        <label class="text-sm opacity-70">Bitiş Tarihi</label>
        <input name="target_date" type="date" value="<?= h(date('Y-m-d', strtotime('+6 months'))) ?>"
               class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
      </div>

      <button class="w-full px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Kaydet</button>
    </form>

    <div class="mt-4 text-xs opacity-70">
      Katkı yaptığında: <b>goal_contributions</b> + <b>transactions(expense)</b> otomatik oluşur.
    </div>
  </div>

  <!-- Goals list -->
  <div class="xl:col-span-2 space-y-4">
    <?php if (!$goals): ?>
      <div class="p-6 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-center opacity-70">
        Henüz hedef yok. Soldan hedef ekleyebilirsin.
      </div>
    <?php endif; ?>

    <?php foreach ($goals as $g): ?>
      <?php
        $goalId = (int)$g['id'];
        $name = (string)$g['name'];
        $target = (float)$g['target_amount'];
        $saved = (float)$g['saved_amount'];
        $status = (string)$g['status'];
        $targetDate = (string)$g['target_date'];

        $pct = ($target > 0) ? max(0, min(100, ($saved / $target) * 100)) : 0;
        $daysLeft = (int)floor((strtotime($targetDate) - strtotime(date('Y-m-d'))) / 86400);
        $daysText = $daysLeft >= 0 ? ($daysLeft . ' gün kaldı') : (abs($daysLeft) . ' gün geçti');

        $badge = 'bg-slate-50 dark:bg-slate-900 border-slate-200 dark:border-slate-800';
        $badgeText = 'Aktif';
        if ($status === 'paused') { $badge='bg-amber-50 dark:bg-amber-950/40 border-amber-200 dark:border-amber-900'; $badgeText='Duraklatıldı'; }
        if ($status === 'completed') { $badge='bg-emerald-50 dark:bg-emerald-950/40 border-emerald-200 dark:border-emerald-900'; $badgeText='Tamamlandı'; }
        if ($status === 'canceled') { $badge='bg-rose-50 dark:bg-rose-950/40 border-rose-200 dark:border-rose-900'; $badgeText='İptal'; }

        $contribSt->execute([$uid, $goalId]);
        $lastContrib = $contribSt->fetchAll();
      ?>

      <div id="goal-<?= $goalId ?>" class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <div class="flex flex-col md:flex-row md:items-start justify-between gap-3">
          <div>
            <div class="flex items-center gap-2">
              <h3 class="text-lg font-semibold"><?= h($name) ?></h3>
              <span class="px-2 py-1 rounded-lg text-xs border <?= $badge ?>"><?= h($badgeText) ?></span>
            </div>

            <div class="mt-1 text-sm opacity-80">
              Hedef: <b><?= number_format($target, 2, ',', '.') ?> ₺</b> • Biriken: <b><?= number_format($saved, 2, ',', '.') ?> ₺</b>
              • Bitiş: <b><?= h($targetDate) ?></b> • <span class="opacity-70"><?= h($daysText) ?></span>
            </div>

            <div class="mt-3 h-3 rounded-full bg-slate-100 dark:bg-slate-900 overflow-hidden border border-slate-200 dark:border-slate-800">
              <div style="width: <?= (float)$pct ?>%;" class="h-full bg-primary"></div>
            </div>
            <div class="mt-2 text-xs opacity-70">%<?= number_format($pct, 1, ',', '.') ?> tamamlandı</div>
          </div>

          <div class="flex flex-wrap gap-2">
            <!-- Add contribution -->
            <button type="button"
              class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm"
              onclick="document.getElementById('contrib-<?= $goalId ?>').classList.toggle('hidden');">
              ➕ Katkı Ekle
            </button>

            <!-- Toggle edit -->
            <button type="button"
              class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm"
              onclick="document.getElementById('edit-<?= $goalId ?>').classList.toggle('hidden');">
              ✏️ Düzenle
            </button>

            <!-- Delete goal -->
            <form method="post" onsubmit="return confirm('Hedef çöp kutusuna taşınsın mı?');">
              <?= csrf_field() ?>
              <input type="hidden" name="action" value="delete_goal">
              <input type="hidden" name="id" value="<?= $goalId ?>">
              <button class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
                🗑️ Sil
              </button>
            </form>
          </div>
        </div>

        <!-- Contribution form -->
        <div id="contrib-<?= $goalId ?>" class="hidden mt-4 p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <h4 class="font-semibold">Katkı Ekle</h4>

          <form class="mt-3 grid grid-cols-1 md:grid-cols-4 gap-3" method="post">
            <?= csrf_field() ?>
            <input type="hidden" name="action" value="add_contribution">
            <input type="hidden" name="goal_id" value="<?= $goalId ?>">

            <div class="md:col-span-2">
              <label class="text-sm opacity-70">Hesap</label>
              <select name="account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800" required>
                <option value="">Seç...</option>
                <?php foreach ($accounts as $a): ?>
                  <option value="<?= (int)$a['id'] ?>"><?= h((string)$a['name']) ?> (<?= h((string)$a['currency']) ?>)</option>
                <?php endforeach; ?>
              </select>
            </div>

            <div>
              <label class="text-sm opacity-70">Tutar</label>
              <input name="amount" type="number" step="0.01" min="0"
                     class="mt-1 w-full px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800" required>
            </div>

            <div>
              <label class="text-sm opacity-70">Tarih</label>
              <input name="contributed_at" type="date" value="<?= h(date('Y-m-d')) ?>"
                     class="mt-1 w-full px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
            </div>

            <div class="md:col-span-4">
              <label class="text-sm opacity-70">Not (ops.)</label>
              <input name="note"
                     class="mt-1 w-full px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800"
                     placeholder="Örn: Ocak katkısı">
            </div>

            <div class="md:col-span-4">
              <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Katkıyı Kaydet</button>
              <span class="text-xs opacity-70 ml-2">Not: Bu kayıt aynı zamanda gider olarak işlenir.</span>
            </div>
          </form>
        </div>

        <!-- Edit form -->
        <div id="edit-<?= $goalId ?>" class="hidden mt-4 p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <h4 class="font-semibold">Hedefi Düzenle</h4>

          <form class="mt-3 grid grid-cols-1 md:grid-cols-4 gap-3" method="post">
            <?= csrf_field() ?>
            <input type="hidden" name="action" value="update_goal">
            <input type="hidden" name="id" value="<?= $goalId ?>">

            <div class="md:col-span-2">
              <label class="text-sm opacity-70">Ad</label>
              <input name="name" value="<?= h($name) ?>"
                     class="mt-1 w-full px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800" required>
            </div>

            <div>
              <label class="text-sm opacity-70">Hedef Tutar</label>
              <input name="target_amount" type="number" step="0.01" min="0" value="<?= h((string)$target) ?>"
                     class="mt-1 w-full px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800" required>
            </div>

            <div>
              <label class="text-sm opacity-70">Bitiş Tarihi</label>
              <input name="target_date" type="date" value="<?= h($targetDate) ?>"
                     class="mt-1 w-full px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800" required>
            </div>

            <div class="md:col-span-2">
              <label class="text-sm opacity-70">Durum</label>
              <select name="status" class="mt-1 w-full px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
                <?php foreach (['active'=>'Aktif','paused'=>'Duraklat','completed'=>'Tamamlandı','canceled'=>'İptal'] as $k=>$label): ?>
                  <option value="<?= h($k) ?>" <?= $status===$k ? 'selected' : '' ?>><?= h($label) ?></option>
                <?php endforeach; ?>
              </select>
            </div>

            <div class="md:col-span-4">
              <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Güncelle</button>
            </div>
          </form>
        </div>

        <!-- Last contributions -->
        <div class="mt-5">
          <div class="flex items-center justify-between">
            <h4 class="font-semibold">Son Katkılar</h4>
            <span class="text-xs opacity-70">Son 5 kayıt</span>
          </div>

          <div class="mt-3 overflow-x-auto">
            <table class="w-full text-sm">
              <thead class="text-xs uppercase opacity-70">
                <tr class="border-b border-slate-200 dark:border-slate-800">
                  <th class="text-left py-2 pr-3">Tarih</th>
                  <th class="text-left py-2 pr-3">Hesap</th>
                  <th class="text-right py-2 pl-3">Tutar</th>
                  <th class="text-left py-2 pr-3">Not</th>
                  <th class="text-right py-2 pl-3">İşlem</th>
                </tr>
              </thead>
              <tbody>
                <?php if (!$lastContrib): ?>
                  <tr><td colspan="5" class="py-6 text-center opacity-70">Katkı yok.</td></tr>
                <?php endif; ?>
                <?php foreach ($lastContrib as $c): ?>
                  <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
                    <td class="py-3 pr-3"><?= h((string)$c['contributed_at']) ?></td>
                    <td class="py-3 pr-3"><?= h((string)($c['account_name'] ?? '—')) ?></td>
                    <td class="py-3 pl-3 text-right"><?= number_format((float)$c['amount'], 2, ',', '.') ?> ₺</td>
                    <td class="py-3 pr-3"><?= h((string)($c['note'] ?? '')) ?></td>
                    <td class="py-3 pl-3 text-right">
                      <form method="post" onsubmit="return confirm('Katkı çöp kutusuna taşınsın mı?');">
                        <?= csrf_field() ?>
                        <input type="hidden" name="action" value="delete_contribution">
                        <input type="hidden" name="id" value="<?= (int)$c['id'] ?>">
                        <input type="hidden" name="goal_id" value="<?= $goalId ?>">
                        <button class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">🗑️</button>
                      </form>
                    </td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>

          <div class="mt-2 text-xs opacity-70">
            Not: Katkıyı silmek şu an sadece <b>goal_contributions</b> kaydını çöp kutusuna taşır.
            İstersen bir sonraki adımda katkıya bağlı <b>transactions(expense)</b> kaydını da eşleştirip birlikte silelim.
          </div>
        </div>

      </div>
    <?php endforeach; ?>
  </div>
</div>

<?php require __DIR__ . '/../inc/footer.php'; ?>
