<?php
declare(strict_types=1);

$type = 'investments';

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';
if (file_exists(__DIR__ . '/../inc/csrf.php')) require __DIR__ . '/../inc/csrf.php';

$uid = function_exists('require_user_id') ? require_user_id($pdo) : (current_user_id() ?? 0);
if ($uid <= 0) { http_response_code(401); echo "unauthorized"; exit; }

$hasCsrf = function_exists('csrf_token') && function_exists('csrf_verify');

$action = trim((string)($_POST['action'] ?? ''));
$csrf   = trim((string)($_POST['csrf'] ?? ''));

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if ($hasCsrf && !csrf_verify($csrf)) {
    flash_set('error', 'Güvenlik doğrulaması başarısız (CSRF). Sayfayı yenileyip tekrar deneyin.');
    header('Location: ' . page_url('investments.php'));
    exit;
  }

  if ($action === 'add') {
    $name     = trim((string)($_POST['name'] ?? ''));
    $platform = trim((string)($_POST['platform'] ?? ''));
    $currency = trim((string)($_POST['currency'] ?? 'TRY'));

    if ($name === '') {
      flash_set('error', 'Yatırım hesabı adı zorunludur.');
      header('Location: ' . page_url('investments.php'));
      exit;
    }

    $st = $pdo->prepare("
      INSERT INTO investment_accounts (user_id, name, platform, currency, is_active, created_at)
      VALUES (?, ?, ?, ?, 1, NOW())
    ");
    $st->execute([$uid, $name, ($platform !== '' ? $platform : null), $currency]);

    flash_set('success', 'Yatırım hesabı eklendi.');
    header('Location: ' . page_url('investments.php'));
    exit;
  }

  if ($action === 'edit') {
    $id       = (int)($_POST['id'] ?? 0);
    $name     = trim((string)($_POST['name'] ?? ''));
    $platform = trim((string)($_POST['platform'] ?? ''));
    $currency = trim((string)($_POST['currency'] ?? 'TRY'));
    $isActive = (int)(((string)($_POST['is_active'] ?? '1')) === '1');

    if ($id <= 0) {
      flash_set('error', 'Geçersiz kayıt.');
      header('Location: ' . page_url('investments.php'));
      exit;
    }
    if ($name === '') {
      flash_set('error', 'Yatırım hesabı adı zorunludur.');
      header('Location: ' . page_url('investments.php'));
      exit;
    }

    $st = $pdo->prepare("
      UPDATE investment_accounts
         SET name=?,
             platform=?,
             currency=?,
             is_active=?
       WHERE id=? AND user_id=? AND deleted_at IS NULL
    ");
    $st->execute([$name, ($platform !== '' ? $platform : null), $currency, $isActive, $id, $uid]);

    flash_set('success', 'Yatırım hesabı güncellendi.');
    header('Location: ' . page_url('investments.php'));
    exit;
  }

  if ($action === 'trash') {
    $id = (int)($_POST['id'] ?? 0);

    if ($id <= 0) {
      flash_set('error', 'Geçersiz kayıt.');
      header('Location: ' . page_url('investments.php'));
      exit;
    }

    $st = $pdo->prepare("UPDATE investment_accounts SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
    $st->execute([$id, $uid]);

    flash_set('success', 'Yatırım hesabı çöp kutusuna taşındı.');
    header('Location: ' . page_url('investments.php'));
    exit;
  }

  flash_set('error', 'Bilinmeyen işlem.');
  header('Location: ' . page_url('investments.php'));
  exit;
}

/* ---- Page data ---- */
$editId  = (int)($_GET['edit'] ?? 0);
$editRow = null;

if ($editId > 0) {
  $st = $pdo->prepare("SELECT * FROM investment_accounts WHERE id=? AND user_id=? AND deleted_at IS NULL");
  $st->execute([$editId, $uid]);
  $editRow = $st->fetch() ?: null;
  if (!$editRow) {
    flash_set('error', 'Düzenlenecek kayıt bulunamadı.');
    header('Location: ' . page_url('investments.php'));
    exit;
  }
}

$listSt = $pdo->prepare("
  SELECT ia.*,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.kind='transfer' AND t.dest_type='investment' AND t.dest_id=ia.id THEN t.amount ELSE 0 END),0) AS cash_in,
    COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.kind='transfer' AND t.source_type='investment' AND t.source_id=ia.id THEN t.amount ELSE 0 END),0) AS cash_out
  FROM investment_accounts ia
  LEFT JOIN transactions t ON t.user_id=ia.user_id
  WHERE ia.user_id=? AND ia.deleted_at IS NULL
  GROUP BY ia.id
  ORDER BY ia.is_active DESC, ia.name ASC
");
$listSt->execute([$uid]);
$rows = $listSt->fetchAll();

require __DIR__ . '/../inc/header.php';
?>

<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
  <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
    <div>
      <h2 class="text-lg font-semibold">📈 Yatırım Hesapları</h2>
      <p class="text-sm opacity-70">Yatırım hesabı ekle, düzenle, çöp kutusuna taşı. Varlık al/sat için diğer sayfaya geç.</p>
    </div>
    <div class="flex flex-wrap gap-2">
      <a class="px-3 py-2 rounded-xl border border-slate-200 dark:border-slate-800 bg-white dark:bg-slate-950 text-sm"
         href="<?= h(page_url('investments_trades.php')) ?>">
        🪙 Varlık & İşlemler
      </a>
    </div>
  </div>
</div>

<?= flash_render(); ?>

<div class="mt-4 grid grid-cols-1 xl:grid-cols-3 gap-4">

  <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h3 class="text-lg font-semibold"><?= $editRow ? '✏️ Yatırım Hesabını Düzenle' : '➕ Yatırım Hesabı Ekle' ?></h3>

    <form method="post" class="mt-4 space-y-3">
      <?php if ($hasCsrf): ?>
        <input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>">
      <?php endif; ?>

      <input type="hidden" name="action" value="<?= $editRow ? 'edit' : 'add' ?>">
      <?php if ($editRow): ?>
        <input type="hidden" name="id" value="<?= (int)$editRow['id'] ?>">
      <?php endif; ?>

      <div>
        <label class="text-sm opacity-70">Hesap Adı</label>
        <input name="name" value="<?= h((string)($editRow['name'] ?? '')) ?>"
               class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
               placeholder="Örn: Binance, Midas, Garanti Yatırım">
      </div>

      <div>
        <label class="text-sm opacity-70">Platform (Opsiyonel)</label>
        <input name="platform" value="<?= h((string)($editRow['platform'] ?? '')) ?>"
               class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
               placeholder="Örn: Binance, İş Yatırım">
      </div>

      <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
        <div>
          <label class="text-sm opacity-70">Para Birimi</label>
          <?php $cur = (string)($editRow['currency'] ?? 'TRY'); ?>
          <select name="currency"
                  class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <?php foreach (['TRY','USD','EUR'] as $c): ?>
              <option value="<?= h($c) ?>" <?= $cur===$c?'selected':'' ?>><?= h($c) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Durum</label>
          <?php $act = (int)($editRow['is_active'] ?? 1); ?>
          <select name="is_active"
                  class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="1" <?= $act===1?'selected':'' ?>>Aktif</option>
            <option value="0" <?= $act===0?'selected':'' ?>>Pasif</option>
          </select>
        </div>
      </div>

      <div class="flex gap-2">
        <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">
          <?= $editRow ? 'Güncelle' : 'Kaydet' ?>
        </button>
        <?php if ($editRow): ?>
          <a class="px-4 py-2 rounded-xl border border-slate-200 dark:border-slate-800"
             href="<?= h(page_url('investments.php')) ?>">İptal</a>
        <?php endif; ?>
      </div>
    </form>
  </div>

  <div class="xl:col-span-2 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <div class="flex items-center justify-between gap-3">
      <h3 class="text-lg font-semibold">📋 Liste</h3>
      <div class="text-sm opacity-70"><?= (int)count($rows) ?> kayıt</div>
    </div>

    <div class="mt-4 overflow-x-auto">
      <table class="w-full text-sm">
        <thead class="text-xs uppercase opacity-70">
          <tr class="border-b border-slate-200 dark:border-slate-800">
            <th class="text-left py-2 pr-3">Hesap</th>
            <th class="text-left py-2 pr-3">Platform</th>
            <th class="text-right py-2 pl-3">Gelen</th>
            <th class="text-right py-2 pl-3">Çıkan</th>
            <th class="text-right py-2 pl-3">Net</th>
            <th class="text-right py-2 pl-3">İşlem</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="6" class="py-10 text-center opacity-70">Henüz yatırım hesabı yok.</td></tr>
          <?php endif; ?>

          <?php foreach ($rows as $r): ?>
            <?php
              $in  = (float)$r['cash_in'];
              $out = (float)$r['cash_out'];
              $net = $in - $out;
            ?>
            <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
              <td class="py-3 pr-3 font-semibold">
                <?= h((string)$r['name']) ?>
                <?php if ((int)$r['is_active'] === 0): ?>
                  <span class="ml-2 text-xs px-2 py-0.5 rounded-full bg-slate-100 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">Pasif</span>
                <?php endif; ?>
              </td>
              <td class="py-3 pr-3"><?= h((string)($r['platform'] ?? '—')) ?></td>
              <td class="py-3 pl-3 text-right"><?= money((float)$in) ?></td>
              <td class="py-3 pl-3 text-right"><?= money((float)$out) ?></td>
              <td class="py-3 pl-3 text-right font-semibold"><?= money((float)$net) ?></td>
              <td class="py-3 pl-3 text-right">
                <a class="text-sm underline mr-3" href="<?= h(page_url('investments.php')) ?>?edit=<?= (int)$r['id'] ?>">Düzenle</a>

                <form method="post" class="inline" onsubmit="return confirm('Bu yatırım hesabı çöp kutusuna taşınacak. Devam?')">
                  <?php if ($hasCsrf): ?>
                    <input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>">
                  <?php endif; ?>
                  <input type="hidden" name="action" value="trash">
                  <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                  <button class="text-sm underline text-red-600">Çöpe Taşı</button>
                </form>
              </td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>

    <div class="mt-3 text-xs opacity-70">
      Çöpe taşınanlar “Çöp Kutusu” sayfasından geri alınabilir veya kalıcı silinebilir.
    </div>
  </div>

</div>

<?php require __DIR__ . '/../inc/footer.php'; ?>
