<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';

$uid = function_exists('require_user_id') ? require_user_id($pdo) : (current_user_id() ?? 0);
if ($uid <= 0) {
  flash_set('error', 'Users tablosunda kullanıcı bulunamadı. Lütfen 1 kullanıcı ekle.');
  header('Location: '.page_url('index.php'));
  exit;
}

function post_str(string $k): string { return trim((string)($_POST[$k] ?? '')); }
function post_int(string $k): int { return (int)($_POST[$k] ?? 0); }
function post_float(string $k): float { return (float)($_POST[$k] ?? 0); }
function h1($v): string { return h((string)$v); }

function valid_currency(string $c): string {
  $c = strtoupper(trim($c));
  return in_array($c, ['TRY','USD','EUR'], true) ? $c : 'TRY';
}

function normalize_symbol(string $s): string {
  $s = strtoupper(trim($s));
  $s = preg_replace('/[^A-Z0-9\-\._]/', '', $s) ?? '';
  return substr($s, 0, 20);
}

$tab = (string)($_GET['tab'] ?? 'portfolio');
$tab = in_array($tab, ['portfolio','add','history'], true) ? $tab : 'portfolio';

$accountsSt = $pdo->prepare("
  SELECT id, name, currency
  FROM investment_accounts
  WHERE user_id=? AND deleted_at IS NULL AND is_active=1
  ORDER BY name
");
$accountsSt->execute([$uid]);
$invAccounts = $accountsSt->fetchAll();

$accountId = (int)($_GET['account_id'] ?? 0);
if ($accountId <= 0 && $invAccounts) $accountId = (int)$invAccounts[0]['id'];

/* -----------------------------
   Actions: add trade / trash trade
------------------------------ */
$action = (string)($_POST['action'] ?? '');
if ($action) {
  try {
    if ($action === 'add_trade') {
      $invId = post_int('investment_account_id');
      $side  = post_str('side');
      $sym   = normalize_symbol(post_str('asset_symbol'));
      $qty   = post_float('quantity');
      $price = post_float('price');
      $fee   = post_float('fee');
      $date  = post_str('traded_at');
      $note  = post_str('note');

      if ($invId <= 0) throw new RuntimeException('Yatırım hesabı seçmelisin.');
      if (!in_array($side, ['buy','sell'], true)) throw new RuntimeException('İşlem türü hatalı.');
      if ($sym === '') throw new RuntimeException('Varlık sembolü zorunlu (örn: BTC, ETH).');
      if ($qty <= 0) throw new RuntimeException('Miktar 0’dan büyük olmalı.');
      if ($price <= 0) throw new RuntimeException('Fiyat 0’dan büyük olmalı.');
      if ($fee < 0) throw new RuntimeException('Komisyon negatif olamaz.');
      if ($date === '') $date = date('Y-m-d');

      // Ensure account belongs to user
      $chk = $pdo->prepare("SELECT id FROM investment_accounts WHERE id=? AND user_id=? AND deleted_at IS NULL LIMIT 1");
      $chk->execute([$invId, $uid]);
      if (!$chk->fetchColumn()) throw new RuntimeException('Yatırım hesabı bulunamadı.');

      $st = $pdo->prepare("
        INSERT INTO investment_trades
          (user_id, investment_account_id, side, asset_symbol, quantity, price, fee, traded_at, note)
        VALUES
          (?, ?, ?, ?, ?, ?, ?, ?, ?)
      ");
      $st->execute([$uid, $invId, $side, $sym, $qty, $price, $fee, $date, ($note !== '' ? $note : null)]);

      flash_set('success', 'Varlık işlemi kaydedildi ✅');
      header('Location: '.page_url('investments_trades.php').'?tab=history&account_id='.$invId);
      exit;
    }

    if ($action === 'trash_trade') {
      $id = (int)($_POST['id'] ?? 0);
      if ($id <= 0) throw new RuntimeException('Geçersiz işlem id.');

      $st = $pdo->prepare("UPDATE investment_trades SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
      $st->execute([$id, $uid]);

      flash_set('success', 'İşlem çöp kutusuna taşındı 🗑️');
      $aid = (int)($_POST['account_id'] ?? 0);
      header('Location: '.page_url('investments_trades.php').'?tab=history&account_id='.$aid);
      exit;
    }

    throw new RuntimeException('Geçersiz işlem.');
  } catch (Throwable $e) {
    flash_set('error', $e->getMessage());
    $aid = (int)($_POST['investment_account_id'] ?? ($_POST['account_id'] ?? $accountId));
    header('Location: '.page_url('investments_trades.php').'?tab='.$tab.'&account_id='.$aid);
    exit;
  }
}

/* -----------------------------
   Portfolio computation (holdings)
------------------------------ */
$holdings = [];
if ($accountId > 0) {
  $st = $pdo->prepare("
    SELECT asset_symbol,
           SUM(CASE WHEN side='buy' THEN quantity ELSE -quantity END) AS qty
    FROM investment_trades
    WHERE user_id=? AND investment_account_id=? AND deleted_at IS NULL
    GROUP BY asset_symbol
    HAVING ABS(qty) > 0.00000001
    ORDER BY asset_symbol
  ");
  $st->execute([$uid, $accountId]);
  $holdings = $st->fetchAll();
}

/* -----------------------------
   History (last 200)
------------------------------ */
$history = [];
if ($accountId > 0) {
  $st = $pdo->prepare("
    SELECT id, side, asset_symbol, quantity, price, fee, traded_at, note
    FROM investment_trades
    WHERE user_id=? AND investment_account_id=? AND deleted_at IS NULL
    ORDER BY traded_at DESC, id DESC
    LIMIT 200
  ");
  $st->execute([$uid, $accountId]);
  $history = $st->fetchAll();
}

/* -----------------------------
   Optional: cash movement summary (transfers to/from investment)
------------------------------ */
$cashSummary = ['in'=>0.0, 'out'=>0.0];
if ($accountId > 0) {
  $st = $pdo->prepare("
    SELECT
      COALESCE(SUM(CASE WHEN kind='transfer' AND dest_type='investment' AND dest_id=? THEN amount ELSE 0 END),0) AS cash_in,
      COALESCE(SUM(CASE WHEN kind='transfer' AND source_type='investment' AND source_id=? THEN amount ELSE 0 END),0) AS cash_out
    FROM transactions
    WHERE user_id=? AND deleted_at IS NULL
  ");
  $st->execute([$accountId, $accountId, $uid]);
  $row = $st->fetch() ?: [];
  $cashSummary['in']  = (float)($row['cash_in'] ?? 0);
  $cashSummary['out'] = (float)($row['cash_out'] ?? 0);
}

require __DIR__ . '/../inc/header.php';
?>

<!-- Chart.js (pie chart) -->
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>

<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
  <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
    <div>
      <h2 class="text-lg font-semibold">📈 Varlık & İşlemler</h2>
      <p class="text-sm opacity-70">Yatırım hesapları, al/sat işlemleri ve portföy dağılımı</p>
    </div>

    <div class="flex flex-wrap gap-2">
      <a href="<?= h1(page_url('trash.php')) ?>?entity=investment_trades"
         class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
        🗑️ Çöp Kutusu
      </a>
      <a href="<?= h1(page_url('investments.php')) ?>"
         class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
        ⚙️ Yatırım Hesapları
      </a>
    </div>
  </div>
</div>

<div class="mt-4 flex flex-col md:flex-row gap-3">
  <div class="md:w-80 p-4 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <label class="text-sm opacity-70">Yatırım Hesabı Seç</label>
    <form method="get" class="mt-2">
      <input type="hidden" name="tab" value="<?= h1($tab) ?>">
      <select name="account_id"
              class="w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
              onchange="this.form.submit()">
        <?php if (!$invAccounts): ?>
          <option value="">Önce yatırım hesabı eklemelisin</option>
        <?php else: ?>
          <?php foreach ($invAccounts as $a): ?>
            <option value="<?= (int)$a['id'] ?>" <?= (int)$a['id']===$accountId ? 'selected' : '' ?>>
              <?= h1($a['name']) ?> (<?= h1($a['currency']) ?>)
            </option>
          <?php endforeach; ?>
        <?php endif; ?>
      </select>
    </form>

    <div class="mt-4 grid grid-cols-2 gap-2 text-sm">
      <div class="p-3 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        <div class="opacity-70 text-xs">Yatırıma Gelen</div>
        <div class="font-semibold"><?= number_format($cashSummary['in'], 2, ',', '.') ?></div>
      </div>
      <div class="p-3 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        <div class="opacity-70 text-xs">Yatırımdan Çıkan</div>
        <div class="font-semibold"><?= number_format($cashSummary['out'], 2, ',', '.') ?></div>
      </div>
    </div>

    <div class="mt-4 flex flex-wrap gap-2">
      <a class="px-3 py-2 rounded-xl border <?= $tab==='portfolio' ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>"
         href="<?= h1(page_url('investments_trades.php')) ?>?tab=portfolio&account_id=<?= (int)$accountId ?>">📊 Portföy</a>
      <a class="px-3 py-2 rounded-xl border <?= $tab==='add' ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>"
         href="<?= h1(page_url('investments_trades.php')) ?>?tab=add&account_id=<?= (int)$accountId ?>">➕ İşlem Ekle</a>
      <a class="px-3 py-2 rounded-xl border <?= $tab==='history' ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>"
         href="<?= h1(page_url('investments_trades.php')) ?>?tab=history&account_id=<?= (int)$accountId ?>">🧾 Geçmiş</a>
    </div>
  </div>

  <div class="flex-1">

    <?php if ($tab === 'portfolio'): ?>
      <div class="grid grid-cols-1 xl:grid-cols-2 gap-4">
        <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
          <h3 class="text-lg font-semibold">📊 Varlık Dağılımı</h3>
          <p class="text-sm opacity-70 mt-1">Miktara göre dağılım (basit). İstersen sonra TRY değere göre yaparız.</p>
          <div class="mt-4">
            <canvas id="pie"></canvas>
          </div>
        </div>

        <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
          <h3 class="text-lg font-semibold">📋 Varlık Listesi</h3>

          <div class="mt-4 overflow-x-auto">
            <table class="w-full text-sm">
              <thead class="text-xs uppercase opacity-70">
                <tr class="border-b border-slate-200 dark:border-slate-800">
                  <th class="text-left py-2 pr-3">Sembol</th>
                  <th class="text-right py-2 pl-3">Miktar</th>
                </tr>
              </thead>
              <tbody>
                <?php if (!$holdings): ?>
                  <tr><td colspan="2" class="py-8 text-center opacity-70">Bu hesapta varlık yok.</td></tr>
                <?php endif; ?>
                <?php foreach ($holdings as $hrow): ?>
                  <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
                    <td class="py-3 pr-3 font-semibold"><?= h1($hrow['asset_symbol']) ?></td>
                    <td class="py-3 pl-3 text-right"><?= rtrim(rtrim(number_format((float)$hrow['qty'], 8, '.', ''), '0'), '.') ?></td>
                  </tr>
                <?php endforeach; ?>
              </tbody>
            </table>
          </div>

          <div class="mt-4 text-xs opacity-70">
            Not: Bu hesaplama sadece <b>investment_trades</b> al/sat işlemlerinden net miktarı çıkarır.
          </div>
        </div>
      </div>

      <script>
        const labels = <?= json_encode(array_map(fn($x)=>(string)$x['asset_symbol'], $holdings), JSON_UNESCAPED_UNICODE) ?>;
        const values = <?= json_encode(array_map(fn($x)=>(float)$x['qty'], $holdings), JSON_UNESCAPED_UNICODE) ?>;

        const ctx = document.getElementById('pie');
        if (ctx && labels.length) {
          new Chart(ctx, {
            type: 'pie',
            data: { labels, datasets: [{ data: values }] },
            options: { plugins: { legend: { position: 'bottom' } } }
          });
        }
      </script>
    <?php endif; ?>

    <?php if ($tab === 'add'): ?>
      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <h3 class="text-lg font-semibold">➕ Varlık İşlemi Ekle</h3>
        <p class="text-sm opacity-70 mt-1">Örn: 1 BTC al / 0.5 ETH sat</p>

        <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
          <input type="hidden" name="action" value="add_trade">
          <input type="hidden" name="investment_account_id" value="<?= (int)$accountId ?>">

          <div>
            <label class="text-sm opacity-70">İşlem Türü</label>
            <select name="side" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
              <option value="buy">Al (BUY)</option>
              <option value="sell">Sat (SELL)</option>
            </select>
          </div>

          <div>
            <label class="text-sm opacity-70">Varlık Sembolü</label>
            <input name="asset_symbol" placeholder="BTC / ETH / XAU / AAPL..."
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
          </div>

          <div>
            <label class="text-sm opacity-70">Miktar</label>
            <input name="quantity" type="number" step="0.00000001" required
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          </div>

          <div>
            <label class="text-sm opacity-70">Fiyat</label>
            <input name="price" type="number" step="0.00000001" required
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          </div>

          <div>
            <label class="text-sm opacity-70">Komisyon (ops.)</label>
            <input name="fee" type="number" step="0.01" value="0"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          </div>

          <div>
            <label class="text-sm opacity-70">Tarih</label>
            <input name="traded_at" type="date" value="<?= h1(date('Y-m-d')) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          </div>

          <div class="md:col-span-2">
            <label class="text-sm opacity-70">Not (ops.)</label>
            <input name="note"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          </div>

          <div class="md:col-span-2 pt-2">
            <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ İşlemi Kaydet</button>
          </div>
        </form>

        <div class="mt-4 text-xs opacity-70">
          İpucu: “Komisyon”u daha sonra raporlarda gider gibi göstermek istersen; bunu ayrıca <b>expense</b> işlemine dönüştürebiliriz.
        </div>
      </div>
    <?php endif; ?>

    <?php if ($tab === 'history'): ?>
      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
        <div class="flex items-center justify-between gap-3">
          <div>
            <h3 class="text-lg font-semibold">🧾 İşlem Geçmişi</h3>
            <p class="text-sm opacity-70 mt-1">Son 200 kayıt</p>
          </div>
        </div>

        <div class="mt-4 overflow-x-auto">
          <table class="w-full text-sm">
            <thead class="text-xs uppercase opacity-70">
              <tr class="border-b border-slate-200 dark:border-slate-800">
                <th class="text-left py-2 pr-3">Tarih</th>
                <th class="text-left py-2 pr-3">Tür</th>
                <th class="text-left py-2 pr-3">Sembol</th>
                <th class="text-right py-2 pl-3">Miktar</th>
                <th class="text-right py-2 pl-3">Fiyat</th>
                <th class="text-right py-2 pl-3">Komisyon</th>
                <th class="text-left py-2 pr-3">Not</th>
                <th class="text-right py-2 pl-3">İşlem</th>
              </tr>
            </thead>
            <tbody>
              <?php if (!$history): ?>
                <tr><td colspan="8" class="py-10 text-center opacity-70">Kayıt yok.</td></tr>
              <?php endif; ?>

              <?php foreach ($history as $r): ?>
                <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
                  <td class="py-3 pr-3"><?= h1($r['traded_at']) ?></td>
                  <td class="py-3 pr-3">
                    <?php if ($r['side'] === 'buy'): ?>
                      <span class="px-2 py-1 rounded-lg text-xs bg-emerald-50 dark:bg-emerald-950/40 border border-emerald-200 dark:border-emerald-900">AL</span>
                    <?php else: ?>
                      <span class="px-2 py-1 rounded-lg text-xs bg-rose-50 dark:bg-rose-950/40 border border-rose-200 dark:border-rose-900">SAT</span>
                    <?php endif; ?>
                  </td>
                  <td class="py-3 pr-3 font-semibold"><?= h1($r['asset_symbol']) ?></td>
                  <td class="py-3 pl-3 text-right"><?= rtrim(rtrim(number_format((float)$r['quantity'], 8, '.', ''), '0'), '.') ?></td>
                  <td class="py-3 pl-3 text-right"><?= rtrim(rtrim(number_format((float)$r['price'], 8, '.', ''), '0'), '.') ?></td>
                  <td class="py-3 pl-3 text-right"><?= number_format((float)$r['fee'], 2, ',', '.') ?></td>
                  <td class="py-3 pr-3"><?= h1($r['note'] ?? '') ?></td>
                  <td class="py-3 pl-3 text-right">
                    <form method="post" onsubmit="return confirm('Çöp kutusuna taşınsın mı?');">
                      <input type="hidden" name="action" value="trash_trade">
                      <input type="hidden" name="id" value="<?= (int)$r['id'] ?>">
                      <input type="hidden" name="account_id" value="<?= (int)$accountId ?>">
                      <button class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
                        🗑️
                      </button>
                    </form>
                  </td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>

        <div class="mt-4 text-xs opacity-70">
          Silme işlemi “kalıcı” değil; <b>deleted_at</b> dolar ve çöp kutusundan geri alınabilir.
        </div>
      </div>
    <?php endif; ?>

  </div>
</div>

<?php require __DIR__ . '/../inc/footer.php'; ?>
