<?php
declare(strict_types=1);

$type = 'notes';

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';
if (file_exists(__DIR__ . '/../inc/csrf.php')) require __DIR__ . '/../inc/csrf.php';

$uid = function_exists('require_user_id') ? require_user_id($pdo) : (current_user_id() ?? 0);
if ($uid <= 0) { http_response_code(401); echo "unauthorized"; exit; }

$hasCsrf = function_exists('csrf_token') && function_exists('csrf_verify');
function post_str(string $k, string $def=''): string { return trim((string)($_POST[$k] ?? $def)); }
function post_int(string $k, int $def=0): int { return (int)($_POST[$k] ?? $def); }
function post_bool(string $k): int { return isset($_POST[$k]) ? 1 : 0; }

$action = post_str('action','');
$csrf   = post_str('csrf','');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if ($hasCsrf && !csrf_verify($csrf)) {
    flash_set('error', 'CSRF doğrulaması başarısız. Sayfayı yenileyip tekrar dene.');
    header('Location: ' . page_url('notes.php'));
    exit;
  }

  if ($action === 'create') {
    $title = post_str('title');
    $content = post_str('content');
    $color = post_str('color','yellow');
    if (!in_array($color, ['yellow','blue','green','pink','purple','gray'], true)) $color = 'yellow';

    $st = $pdo->prepare("INSERT INTO notes (user_id,title,content,color,is_pinned,sort_order) VALUES (?,?,?,?,0,0)");
    $st->execute([$uid, $title ?: null, $content ?: null, $color]);

    flash_set('success', 'Not eklendi.');
    header('Location: ' . page_url('notes.php'));
    exit;
  }

  if ($action === 'update') {
    $id = post_int('id');
    $title = post_str('title');
    $content = post_str('content');
    $color = post_str('color','yellow');
    $pin = post_bool('is_pinned');

    if (!in_array($color, ['yellow','blue','green','pink','purple','gray'], true)) $color = 'yellow';

    $st = $pdo->prepare("
      UPDATE notes
         SET title=?, content=?, color=?, is_pinned=?, updated_at=NOW()
       WHERE id=? AND user_id=? AND deleted_at IS NULL
    ");
    $st->execute([$title ?: null, $content ?: null, $color, $pin, $id, $uid]);

    flash_set('success', 'Not güncellendi.');
    header('Location: ' . page_url('notes.php'));
    exit;
  }

  if ($action === 'delete') {
    $id = post_int('id');
    $st = $pdo->prepare("UPDATE notes SET deleted_at=NOW() WHERE id=? AND user_id=? AND deleted_at IS NULL");
    $st->execute([$id,$uid]);
    flash_set('success', 'Not çöp kutusuna taşındı.');
    header('Location: ' . page_url('notes.php'));
    exit;
  }

  if ($action === 'restore') {
    $id = post_int('id');
    $st = $pdo->prepare("UPDATE notes SET deleted_at=NULL WHERE id=? AND user_id=? AND deleted_at IS NOT NULL");
    $st->execute([$id,$uid]);
    flash_set('success', 'Not geri alındı.');
    header('Location: ' . page_url('notes.php') . '?trash=1');
    exit;
  }

  if ($action === 'purge') {
    $id = post_int('id');
    $st = $pdo->prepare("DELETE FROM notes WHERE id=? AND user_id=? AND deleted_at IS NOT NULL");
    $st->execute([$id,$uid]);
    flash_set('success', 'Not kalıcı silindi.');
    header('Location: ' . page_url('notes.php') . '?trash=1');
    exit;
  }

  if ($action === 'set_view') {
    $mode = post_str('view_mode','sticky');
    if (!in_array($mode, ['sticky','list'], true)) $mode = 'sticky';
    // kullanıcı bazlı saklamak için settings'e yazabiliriz; şimdilik query ile
    header('Location: ' . page_url('notes.php') . '?view=' . urlencode($mode));
    exit;
  }

  flash_set('error', 'Bilinmeyen işlem.');
  header('Location: ' . page_url('notes.php'));
  exit;
}

$trash = ((int)($_GET['trash'] ?? 0) === 1);
$view = (string)($_GET['view'] ?? 'sticky');
if (!in_array($view, ['sticky','list'], true)) $view = 'sticky';

if ($trash) {
  $st = $pdo->prepare("SELECT * FROM notes WHERE user_id=? AND deleted_at IS NOT NULL ORDER BY deleted_at DESC");
  $st->execute([$uid]);
} else {
  $st = $pdo->prepare("SELECT * FROM notes WHERE user_id=? AND deleted_at IS NULL ORDER BY is_pinned DESC, sort_order ASC, updated_at DESC, created_at DESC");
  $st->execute([$uid]);
}
$notes = $st->fetchAll(PDO::FETCH_ASSOC);

require __DIR__ . '/../inc/header.php';
?>

<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
  <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
    <div>
      <h2 class="text-lg font-semibold">📝 Notlar</h2>
      <p class="text-sm opacity-70">Yapışkan not gibi hızlı notlar. İstersen liste görünümüne de geç.</p>
    </div>

    <div class="flex items-center gap-2">
      <a class="px-3 py-2 rounded-xl text-sm border border-slate-200 dark:border-slate-800"
         href="<?= h(page_url('notes.php')) ?>?view=sticky">🟨 Sticky</a>
      <a class="px-3 py-2 rounded-xl text-sm border border-slate-200 dark:border-slate-800"
         href="<?= h(page_url('notes.php')) ?>?view=list">📋 Liste</a>

      <a class="px-3 py-2 rounded-xl text-sm border border-slate-200 dark:border-slate-800"
         href="<?= h(page_url('notes.php')) ?><?= $trash ? '' : '?trash=1' ?>">
        <?= $trash ? '⬅️ Notlara Dön' : '🗑️ Çöp' ?>
      </a>
    </div>
  </div>
</div>

<?= function_exists('flash_render') ? flash_render() : '' ?>

<div class="mt-4 grid grid-cols-1 xl:grid-cols-3 gap-4">
  <div class="xl:col-span-1 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h3 class="text-lg font-semibold"><?= $trash ? '🗑️ Çöp Kutusu' : '➕ Yeni Not' ?></h3>

    <?php if (!$trash): ?>
      <form method="post" class="mt-4 space-y-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="create">

        <div>
          <label class="text-sm opacity-70">Başlık</label>
          <input name="title" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" placeholder="Örn: Fatura günü">
        </div>

        <div>
          <label class="text-sm opacity-70">Not</label>
          <textarea name="content" rows="5" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" placeholder="Örn: 15’inde kart kesim..."></textarea>
        </div>

        <div>
          <label class="text-sm opacity-70">Renk</label>
          <select name="color" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="yellow">Sarı</option>
            <option value="blue">Mavi</option>
            <option value="green">Yeşil</option>
            <option value="pink">Pembe</option>
            <option value="purple">Mor</option>
            <option value="gray">Gri</option>
          </select>
        </div>

        <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Not Ekle</button>
      </form>
    <?php else: ?>
      <p class="text-sm opacity-70 mt-2">Silinen notlar burada. Geri alabilir veya kalıcı silebilirsin.</p>
    <?php endif; ?>
  </div>

  <div class="xl:col-span-2 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <div class="flex items-center justify-between">
      <h3 class="text-lg font-semibold"><?= $trash ? '🗑️ Silinen Notlar' : '📌 Notların' ?></h3>
      <div class="text-sm opacity-70"><?= $view === 'sticky' ? 'Sticky görünüm' : 'Liste görünüm' ?></div>
    </div>

    <?php if (!$notes): ?>
      <div class="mt-4 opacity-70">Henüz not yok.</div>
    <?php else: ?>

      <?php if ($view === 'list'): ?>
        <div class="mt-4 space-y-3">
          <?php foreach ($notes as $n): ?>
            <div class="p-4 rounded-2xl border border-slate-200 dark:border-slate-800 bg-slate-50 dark:bg-slate-900">
              <div class="flex items-center gap-2">
                <div class="font-semibold"><?= h((string)($n['title'] ?? 'Not')) ?></div>
                <?php if ((int)($n['is_pinned'] ?? 0) === 1): ?><span class="text-xs px-2 py-1 rounded-lg bg-slate-900 text-white dark:bg-white dark:text-slate-900">PIN</span><?php endif; ?>
                <div class="ml-auto text-xs opacity-70"><?= h((string)($n['updated_at'] ?? $n['created_at'] ?? '')) ?></div>
              </div>
              <div class="mt-2 text-sm whitespace-pre-wrap"><?= h((string)($n['content'] ?? '')) ?></div>

              <div class="mt-3 flex items-center gap-2">
                <?php if (!$trash): ?>
                  <button class="px-3 py-2 rounded-xl text-sm border border-slate-200 dark:border-slate-800"
                          type="button"
                          onclick="openEdit(<?= (int)$n['id'] ?>, <?= json_encode((string)($n['title'] ?? '')) ?>, <?= json_encode((string)($n['content'] ?? '')) ?>, <?= json_encode((string)($n['color'] ?? 'yellow')) ?>, <?= (int)($n['is_pinned'] ?? 0) ?>)">Düzenle</button>

                  <form method="post">
                    <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" value="<?= (int)$n['id'] ?>">
                    <button class="px-3 py-2 rounded-xl text-sm border border-slate-200 dark:border-slate-800">Çöpe At</button>
                  </form>
                <?php else: ?>
                  <form method="post">
                    <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
                    <input type="hidden" name="action" value="restore">
                    <input type="hidden" name="id" value="<?= (int)$n['id'] ?>">
                    <button class="px-3 py-2 rounded-xl text-sm border border-slate-200 dark:border-slate-800">Geri Al</button>
                  </form>
                  <form method="post" onsubmit="return confirm('Kalıcı silinsin mi?');">
                    <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
                    <input type="hidden" name="action" value="purge">
                    <input type="hidden" name="id" value="<?= (int)$n['id'] ?>">
                    <button class="px-3 py-2 rounded-xl text-sm border border-rose-200 text-rose-600 dark:border-rose-900 dark:text-rose-300">Kalıcı Sil</button>
                  </form>
                <?php endif; ?>
              </div>
            </div>
          <?php endforeach; ?>
        </div>

      <?php else: ?>
        <!-- Sticky view -->
        <style>
          .note-yellow{ background: #fff7cc; }
          .note-blue{ background: #dbeafe; }
          .note-green{ background: #dcfce7; }
          .note-pink{ background: #fce7f3; }
          .note-purple{ background: #ede9fe; }
          .note-gray{ background: #e5e7eb; }
          .dark .note-yellow{ background: #3a3216; }
          .dark .note-blue{ background: #0b223f; }
          .dark .note-green{ background: #0d2b19; }
          .dark .note-pink{ background: #3a0d25; }
          .dark .note-purple{ background: #1f173a; }
          .dark .note-gray{ background: #111827; }
        </style>

        <div class="mt-4 grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
          <?php foreach ($notes as $n): $c = (string)($n['color'] ?? 'yellow'); ?>
            <div class="p-4 rounded-2xl border border-slate-200/70 dark:border-slate-800 note-<?= h($c) ?>">
              <div class="flex items-start gap-2">
                <div class="font-semibold text-base"><?= h((string)($n['title'] ?? 'Not')) ?></div>
                <?php if ((int)($n['is_pinned'] ?? 0) === 1): ?>
                  <span class="text-xs px-2 py-1 rounded-lg bg-slate-900 text-white dark:bg-white dark:text-slate-900">📌</span>
                <?php endif; ?>
                <div class="ml-auto text-xs opacity-70"><?= h((string)($n['updated_at'] ?? $n['created_at'] ?? '')) ?></div>
              </div>

              <div class="mt-2 text-sm whitespace-pre-wrap"><?= h((string)($n['content'] ?? '')) ?></div>

              <div class="mt-3 flex items-center gap-2">
                <?php if (!$trash): ?>
                  <button class="px-3 py-2 rounded-xl text-sm border border-slate-200/60 dark:border-slate-800"
                          type="button"
                          onclick="openEdit(<?= (int)$n['id'] ?>, <?= json_encode((string)($n['title'] ?? '')) ?>, <?= json_encode((string)($n['content'] ?? '')) ?>, <?= json_encode((string)($n['color'] ?? 'yellow')) ?>, <?= (int)($n['is_pinned'] ?? 0) ?>)">Düzenle</button>

                  <form method="post">
                    <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
                    <input type="hidden" name="action" value="delete">
                    <input type="hidden" name="id" value="<?= (int)$n['id'] ?>">
                    <button class="px-3 py-2 rounded-xl text-sm border border-slate-200/60 dark:border-slate-800">Çöpe At</button>
                  </form>
                <?php else: ?>
                  <form method="post">
                    <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
                    <input type="hidden" name="action" value="restore">
                    <input type="hidden" name="id" value="<?= (int)$n['id'] ?>">
                    <button class="px-3 py-2 rounded-xl text-sm border border-slate-200/60 dark:border-slate-800">Geri Al</button>
                  </form>
                  <form method="post" onsubmit="return confirm('Kalıcı silinsin mi?');">
                    <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
                    <input type="hidden" name="action" value="purge">
                    <input type="hidden" name="id" value="<?= (int)$n['id'] ?>">
                    <button class="px-3 py-2 rounded-xl text-sm border border-rose-200 text-rose-700 dark:border-rose-900 dark:text-rose-300">Kalıcı Sil</button>
                  </form>
                <?php endif; ?>
              </div>
            </div>
          <?php endforeach; ?>
        </div>
      <?php endif; ?>

    <?php endif; ?>
  </div>
</div>

<!-- Edit modal -->
<div id="editModal" class="hidden fixed inset-0 z-50 items-center justify-center p-4">
  <div class="absolute inset-0 bg-black/40" onclick="closeEdit()"></div>
  <div class="relative w-full max-w-xl p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <div class="flex items-center justify-between">
      <h3 class="text-lg font-semibold">✏️ Not Düzenle</h3>
      <button class="text-sm opacity-70" onclick="closeEdit()">Kapat</button>
    </div>

    <form method="post" class="mt-4 space-y-3">
      <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
      <input type="hidden" name="action" value="update">
      <input type="hidden" name="id" id="e_id" value="0">

      <div>
        <label class="text-sm opacity-70">Başlık</label>
        <input name="title" id="e_title" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
      </div>

      <div>
        <label class="text-sm opacity-70">Not</label>
        <textarea name="content" id="e_content" rows="5" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"></textarea>
      </div>

      <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
        <div>
          <label class="text-sm opacity-70">Renk</label>
          <select name="color" id="e_color" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="yellow">Sarı</option>
            <option value="blue">Mavi</option>
            <option value="green">Yeşil</option>
            <option value="pink">Pembe</option>
            <option value="purple">Mor</option>
            <option value="gray">Gri</option>
          </select>
        </div>
        <label class="flex items-center gap-3 p-3 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <input type="checkbox" name="is_pinned" id="e_pin">
          <span class="font-semibold">Sabitle (Pin)</span>
        </label>
      </div>

      <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Kaydet</button>
    </form>
  </div>
</div>

<script>
  function openEdit(id, title, content, color, pinned){
    const m = document.getElementById('editModal');
    document.getElementById('e_id').value = id;
    document.getElementById('e_title').value = title || '';
    document.getElementById('e_content').value = content || '';
    document.getElementById('e_color').value = color || 'yellow';
    document.getElementById('e_pin').checked = (Number(pinned) === 1);
    m.classList.remove('hidden');
    m.classList.add('flex');
  }
  function closeEdit(){
    const m = document.getElementById('editModal');
    m.classList.add('hidden');
    m.classList.remove('flex');
  }
</script>

<?php require __DIR__ . '/../inc/footer.php'; ?>
