<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';
if (file_exists(__DIR__ . '/../inc/csrf.php')) require __DIR__ . '/../inc/csrf.php';

$uid = function_exists('require_user_id') ? require_user_id($pdo) : (current_user_id() ?? 0);
if ($uid <= 0) {
  flash_set('error', 'Users tablosunda kullanıcı bulunamadı. Lütfen 1 kullanıcı ekle.');
  header('Location: ' . page_url('index.php'));
  exit;
}

function qstr(string $k, string $def=''): string { return trim((string)($_GET[$k] ?? $def)); }
function qint(string $k, int $def=0): int { return (int)($_GET[$k] ?? $def); }
function h1($v): string { return h((string)$v); }

$tab = qstr('tab', 'summary');
$tab = in_array($tab, ['summary','income','expense','cards','investments','goals'], true) ? $tab : 'summary';

$today = date('Y-m-d');
$from = qstr('from', date('Y-m-01'));
$to   = qstr('to', $today);

// quick month/year
$month = qint('month', 0); // 1-12
$year  = qint('year', 0);

if ($month >= 1 && $month <= 12 && $year >= 2000) {
  $from = sprintf('%04d-%02d-01', $year, $month);
  $to   = date('Y-m-t', strtotime($from));
}

// print/pdf mode
$isPrint = qint('print', 0) === 1;

// optional filters
$kind = qstr('kind', ''); // income/expense/transfer
if ($kind !== '' && !in_array($kind, ['income','expense','transfer'], true)) $kind = '';

$categoryId = qint('category_id', 0);
$accountId  = qint('account_id', 0);
$cardId     = qint('card_id', 0);
$invId      = qint('investment_account_id', 0);

function date_ok(string $d): bool {
  return (bool)preg_match('/^\d{4}-\d{2}-\d{2}$/', $d);
}
if (!date_ok($from)) $from = date('Y-m-01');
if (!date_ok($to))   $to   = $today;
if ($from > $to) { $tmp=$from; $from=$to; $to=$tmp; }

$generatedAt = date('Y-m-d H:i');

// lookups for filters
$cats = $pdo->prepare("SELECT id, type, name FROM categories WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY type, sort_order, name");
$cats->execute([$uid]);
$categories = $cats->fetchAll();

$accs = $pdo->prepare("SELECT id,name,currency FROM accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$accs->execute([$uid]);
$accounts = $accs->fetchAll();

$cardsSt = $pdo->prepare("SELECT id,name,currency,card_limit,opening_debt FROM credit_cards WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$cardsSt->execute([$uid]);
$cards = $cardsSt->fetchAll();

$invSt = $pdo->prepare("SELECT id,name,currency FROM investment_accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$invSt->execute([$uid]);
$investments = $invSt->fetchAll();

/* -----------------------------
  Helpers for “Detay” field
------------------------------ */
function name_by_id(array $rows, int $id): string {
  foreach ($rows as $r) if ((int)$r['id'] === $id) return (string)$r['name'];
  return '—';
}
function category_name(array $cats, ?int $id): string {
  if (!$id) return '—';
  foreach ($cats as $c) if ((int)$c['id'] === $id) return (string)$c['name'];
  return '—';
}
function fmt_money(float $v): string {
  $sign = $v < 0 ? '-' : '';
  $v = abs($v);
  return $sign . '₺' . number_format($v, 2, ',', '.');
}
function fmt_money_signed(string $kind, float $amount): string {
  // income + , expense -
  if ($kind === 'income') return '+'.fmt_money($amount);
  if ($kind === 'expense') return '-'.fmt_money($amount);
  return fmt_money($amount);
}

function build_detail(array $row, array $cats, array $accs, array $cards, array $invs): string {
  $kind = (string)$row['kind'];
  $cat  = category_name($cats, $row['category_id'] ? (int)$row['category_id'] : null);

  $srcT = (string)$row['source_type'];
  $dstT = (string)$row['dest_type'];
  $srcId = $row['source_id'] ? (int)$row['source_id'] : 0;
  $dstId = $row['dest_id'] ? (int)$row['dest_id'] : 0;

  $srcName = '—';
  $dstName = '—';

  if ($srcT === 'account') $srcName = name_by_id($accs, $srcId);
  if ($srcT === 'credit_card') $srcName = name_by_id($cards, $srcId);
  if ($srcT === 'investment') $srcName = name_by_id($invs, $srcId);

  if ($dstT === 'account') $dstName = name_by_id($accs, $dstId);
  if ($dstT === 'credit_card') $dstName = name_by_id($cards, $dstId);
  if ($dstT === 'investment') $dstName = name_by_id($invs, $dstId);

  $desc = (string)($row['description'] ?? '');

  if ($kind === 'transfer') {
    $base = "Transfer • {$srcName} → {$dstName}";
    return $desc !== '' ? ($base . " — " . $desc) : $base;
  }

  // income/expense
  $where = '—';
  if ($srcT !== 'none') $where = $srcName;
  if ($dstT !== 'none' && $srcT === 'none') $where = $dstName;

  $base = "{$cat} • {$where}";
  return $desc !== '' ? ($base . " — " . $desc) : $base;
}

/* -----------------------------
  Summary cards (range)
------------------------------ */
$sumSt = $pdo->prepare("
  SELECT
    COALESCE(SUM(CASE WHEN kind='income'  THEN amount ELSE 0 END),0) AS income_sum,
    COALESCE(SUM(CASE WHEN kind='expense' THEN amount ELSE 0 END),0) AS expense_sum,
    COALESCE(SUM(CASE WHEN kind='transfer' THEN amount ELSE 0 END),0) AS transfer_sum
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
    AND occurred_on BETWEEN ? AND ?
");
$sumSt->execute([$uid, $from, $to]);
$sum = $sumSt->fetch() ?: ['income_sum'=>0,'expense_sum'=>0,'transfer_sum'=>0];

$incomeSum = (float)$sum['income_sum'];
$expenseSum = (float)$sum['expense_sum'];
$netSum = $incomeSum - $expenseSum;

/* -----------------------------
  “As of to” totals for cards/investments
  (cumulative <= $to)
------------------------------ */
$cardDebtRows = [];
if ($cards) {
  $cardDebtSt = $pdo->prepare("
    SELECT
      c.id,
      c.name,
      c.currency,
      c.card_limit,
      c.opening_debt,
      COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.kind='expense' AND t.source_type='credit_card' AND t.source_id=c.id THEN t.amount ELSE 0 END),0) AS card_expenses,
      COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.kind='transfer' AND t.dest_type='credit_card' AND t.dest_id=c.id THEN t.amount ELSE 0 END),0) AS payments_to_card,
      COALESCE(SUM(CASE WHEN t.deleted_at IS NULL AND t.kind='transfer' AND t.source_type='credit_card' AND t.source_id=c.id THEN t.amount ELSE 0 END),0) AS card_out_transfers
    FROM credit_cards c
    LEFT JOIN transactions t
      ON t.user_id=c.user_id
     AND t.occurred_on <= ?
    WHERE c.user_id=? AND c.deleted_at IS NULL AND c.is_active=1
    GROUP BY c.id
    ORDER BY c.name
  ");
  $cardDebtSt->execute([$to, $uid]);
  $cardDebtRows = $cardDebtSt->fetchAll();
}

$totalCardDebt = 0.0;
foreach ($cardDebtRows as $r) {
  $opening = (float)$r['opening_debt'];
  $exp = (float)$r['card_expenses'];
  $payments = (float)$r['payments_to_card'];
  $out = (float)$r['card_out_transfers']; // cash advance etc increases debt
  $debt = $opening + $exp + $out - $payments;
  $totalCardDebt += $debt;
}

/* Investments cash in/out as of to */
$invCash = ['in'=>0.0,'out'=>0.0];
$invCashSt = $pdo->prepare("
  SELECT
    COALESCE(SUM(CASE WHEN kind='transfer' AND dest_type='investment' THEN amount ELSE 0 END),0) AS cash_in,
    COALESCE(SUM(CASE WHEN kind='transfer' AND source_type='investment' THEN amount ELSE 0 END),0) AS cash_out
  FROM transactions
  WHERE user_id=? AND deleted_at IS NULL
    AND occurred_on <= ?
");
$invCashSt->execute([$uid, $to]);
$invCashRow = $invCashSt->fetch() ?: ['cash_in'=>0,'cash_out'=>0];
$invCash['in'] = (float)$invCashRow['cash_in'];
$invCash['out'] = (float)$invCashRow['cash_out'];
$invNet = $invCash['in'] - $invCash['out'];

/* -----------------------------
  Transaction list (filtered)
------------------------------ */
$where = ["t.user_id=?","t.deleted_at IS NULL","t.occurred_on BETWEEN ? AND ?"];
$params = [$uid, $from, $to];

if ($kind !== '') { $where[]="t.kind=?"; $params[]=$kind; }
if ($categoryId > 0) { $where[]="t.category_id=?"; $params[]=$categoryId; }

if ($accountId > 0) {
  $where[] = "(
      (t.source_type='account' AND t.source_id=?)
   OR (t.dest_type='account' AND t.dest_id=?)
  )";
  $params[]=$accountId; $params[]=$accountId;
}
if ($cardId > 0) {
  $where[] = "(
      (t.source_type='credit_card' AND t.source_id=?)
   OR (t.dest_type='credit_card' AND t.dest_id=?)
  )";
  $params[]=$cardId; $params[]=$cardId;
}
if ($invId > 0) {
  $where[] = "(
      (t.source_type='investment' AND t.source_id=?)
   OR (t.dest_type='investment' AND t.dest_id=?)
  )";
  $params[]=$invId; $params[]=$invId;
}

$sqlList = "
  SELECT
    t.id, t.occurred_on, t.kind, t.amount, t.currency, t.category_id, t.description,
    t.source_type, t.source_id, t.dest_type, t.dest_id
  FROM transactions t
  WHERE ".implode(" AND ", $where)."
  ORDER BY t.occurred_on DESC, t.id DESC
  LIMIT 2000
";
$listSt = $pdo->prepare($sqlList);
$listSt->execute($params);
$rows = $listSt->fetchAll();

/* -----------------------------
  Breakdown tables (top categories)
------------------------------ */
$topIncomeCats = [];
$topExpenseCats = [];

$topSt = $pdo->prepare("
  SELECT c.name, COALESCE(SUM(t.amount),0) AS total
  FROM transactions t
  LEFT JOIN categories c ON c.id=t.category_id
  WHERE t.user_id=? AND t.deleted_at IS NULL
    AND t.occurred_on BETWEEN ? AND ?
    AND t.kind=?
  GROUP BY c.name
  ORDER BY total DESC
  LIMIT 10
");
$topSt->execute([$uid, $from, $to, 'income']);
$topIncomeCats = $topSt->fetchAll();

$topSt->execute([$uid, $from, $to, 'expense']);
$topExpenseCats = $topSt->fetchAll();

/* Goals in range */
$goalRange = [];
$goalSt = $pdo->prepare("
  SELECT g.id, g.name,
         COALESCE(SUM(gc.amount),0) AS total
  FROM goal_contributions gc
  LEFT JOIN goals g ON g.id=gc.goal_id
  WHERE gc.user_id=? AND gc.deleted_at IS NULL
    AND gc.contributed_at BETWEEN ? AND ?
  GROUP BY g.id, g.name
  ORDER BY total DESC
");
$goalSt->execute([$uid, $from, $to]);
$goalRange = $goalSt->fetchAll();

$periodText = "{$from} - {$to}";

require __DIR__ . '/../inc/header.php';
?>

<?php if ($isPrint): ?>
  <style>
    @media print {
      .no-print { display:none !important; }
      body { background: #fff !important; }
      .print-wrap { padding: 0 !important; }
      .card { border: 1px solid #ddd !important; }
    }
  </style>
<?php endif; ?>

<div class="print-wrap">

  <!-- Header -->
  <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
    <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
      <div>
        <h2 class="text-lg font-semibold">📊 Raporlar & Analiz</h2>
        <p class="text-sm opacity-70">Filtrele, detayları gör, PDF al (Yazdır).</p>
      </div>

      <div class="no-print flex flex-wrap gap-2">
  
		<button type="button"
		  class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm"
		  onclick="window.print()">
		  🖨️ Yazdır / PDF
		</button>
      </div>
    </div>
  </div>

  <!-- Filters -->
  <div class="no-print mt-4 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
    <form method="get" class="grid grid-cols-1 md:grid-cols-12 gap-3">
      <input type="hidden" name="tab" value="<?= h1($tab) ?>">

      <div class="md:col-span-3">
        <label class="text-sm opacity-70">Başlangıç</label>
        <input type="date" name="from" value="<?= h1($from) ?>"
               class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
      </div>

      <div class="md:col-span-3">
        <label class="text-sm opacity-70">Bitiş</label>
        <input type="date" name="to" value="<?= h1($to) ?>"
               class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
      </div>

      <div class="md:col-span-2">
        <label class="text-sm opacity-70">Ay</label>
        <select name="month" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <option value="0">Seç (ops.)</option>
          <?php for($m=1;$m<=12;$m++): ?>
            <option value="<?= $m ?>" <?= ($month===$m ? 'selected':'') ?>><?= $m ?></option>
          <?php endfor; ?>
        </select>
      </div>

      <div class="md:col-span-2">
        <label class="text-sm opacity-70">Yıl</label>
        <input type="number" name="year" value="<?= $year ?: (int)date('Y') ?>"
               class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" min="2000" max="2100">
      </div>

      <div class="md:col-span-2 flex items-end gap-2">
        <button class="w-full px-4 py-2 rounded-xl bg-primary text-white font-semibold">Uygula</button>
      </div>

      <div class="md:col-span-12 grid grid-cols-1 md:grid-cols-12 gap-3 pt-2">
        <div class="md:col-span-3">
          <label class="text-sm opacity-70">Tür</label>
          <select name="kind" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="">Hepsi</option>
            <option value="income" <?= $kind==='income'?'selected':'' ?>>Gelir</option>
            <option value="expense" <?= $kind==='expense'?'selected':'' ?>>Gider</option>
            <option value="transfer" <?= $kind==='transfer'?'selected':'' ?>>Transfer</option>
          </select>
        </div>

        <div class="md:col-span-3">
          <label class="text-sm opacity-70">Kategori</label>
          <select name="category_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="0">Hepsi</option>
            <?php foreach ($categories as $c): ?>
              <?php if (in_array((string)$c['type'], ['income','expense'], true)): ?>
                <option value="<?= (int)$c['id'] ?>" <?= $categoryId===(int)$c['id']?'selected':'' ?>>
                  <?= h1($c['type']) ?> • <?= h1($c['name']) ?>
                </option>
              <?php endif; ?>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Hesap</label>
          <select name="account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="0">Hepsi</option>
            <?php foreach ($accounts as $a): ?>
              <option value="<?= (int)$a['id'] ?>" <?= $accountId===(int)$a['id']?'selected':'' ?>><?= h1($a['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Kredi Kartı</label>
          <select name="card_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="0">Hepsi</option>
            <?php foreach ($cards as $c): ?>
              <option value="<?= (int)$c['id'] ?>" <?= $cardId===(int)$c['id']?'selected':'' ?>><?= h1($c['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Yatırım Hesabı</label>
          <select name="investment_account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="0">Hepsi</option>
            <?php foreach ($investments as $i): ?>
              <option value="<?= (int)$i['id'] ?>" <?= $invId===(int)$i['id']?'selected':'' ?>><?= h1($i['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>
      </div>

    </form>
  </div>

  <!-- Tabs -->
  <div class="mt-4 no-print flex flex-wrap gap-2">
    <?php
      $tabs = [
        'summary' => 'Genel Özet',
        'income' => 'Gelir',
        'expense' => 'Gider',
        'cards' => 'Kartlar',
        'investments' => 'Yatırımlar',
        'goals' => 'Hedefler',
      ];
      foreach ($tabs as $k=>$label):
        $active = $tab===$k;
        $qs = array_merge($_GET, ['tab'=>$k]);
    ?>
      <a href="<?= h1(page_url('reports.php')) ?>?<?= http_build_query($qs) ?>"
         class="px-3 py-2 rounded-xl border text-sm <?= $active ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>">
        <?= h1($label) ?>
      </a>
    <?php endforeach; ?>
  </div>

  <!-- Report title -->
  <div class="mt-4 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
    <div class="text-sm opacity-80">
      Bu rapor <b><?= h1($generatedAt) ?></b> tarihinde oluşturulmuştur.
      <b><?= h1($periodText) ?></b> dönemini göstermektedir.
    </div>
  </div>

  <!-- Summary cards -->
  <div class="mt-4 grid grid-cols-1 md:grid-cols-3 gap-4">
    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
      <div class="text-sm opacity-70">Gelir</div>
      <div class="text-2xl font-semibold mt-1"><?= fmt_money($incomeSum) ?></div>
    </div>
    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
      <div class="text-sm opacity-70">Gider</div>
      <div class="text-2xl font-semibold mt-1"><?= fmt_money($expenseSum) ?></div>
    </div>
    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
      <div class="text-sm opacity-70">Net</div>
      <div class="text-2xl font-semibold mt-1"><?= fmt_money($netSum) ?></div>
    </div>
  </div>

  <?php if ($tab === 'summary'): ?>
    <div class="mt-4 grid grid-cols-1 xl:grid-cols-2 gap-4">
      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
        <h3 class="text-lg font-semibold">🏷️ En Çok Gelir Kategorileri</h3>
        <div class="mt-3 overflow-x-auto">
          <table class="w-full text-sm">
            <thead class="text-xs uppercase opacity-70">
              <tr class="border-b border-slate-200 dark:border-slate-800">
                <th class="text-left py-2 pr-3">Kategori</th>
                <th class="text-right py-2 pl-3">Tutar</th>
              </tr>
            </thead>
            <tbody>
              <?php if (!$topIncomeCats): ?>
                <tr><td colspan="2" class="py-6 text-center opacity-70">Veri yok.</td></tr>
              <?php endif; ?>
              <?php foreach ($topIncomeCats as $r): ?>
                <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
                  <td class="py-3 pr-3"><?= h1($r['name'] ?? '—') ?></td>
                  <td class="py-3 pl-3 text-right"><?= fmt_money((float)$r['total']) ?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>

      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
        <h3 class="text-lg font-semibold">🧾 En Çok Gider Kategorileri</h3>
        <div class="mt-3 overflow-x-auto">
          <table class="w-full text-sm">
            <thead class="text-xs uppercase opacity-70">
              <tr class="border-b border-slate-200 dark:border-slate-800">
                <th class="text-left py-2 pr-3">Kategori</th>
                <th class="text-right py-2 pl-3">Tutar</th>
              </tr>
            </thead>
            <tbody>
              <?php if (!$topExpenseCats): ?>
                <tr><td colspan="2" class="py-6 text-center opacity-70">Veri yok.</td></tr>
              <?php endif; ?>
              <?php foreach ($topExpenseCats as $r): ?>
                <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
                  <td class="py-3 pr-3"><?= h1($r['name'] ?? '—') ?></td>
                  <td class="py-3 pl-3 text-right"><?= fmt_money((float)$r['total']) ?></td>
                </tr>
              <?php endforeach; ?>
            </tbody>
          </table>
        </div>
      </div>
    </div>

    <div class="mt-4 grid grid-cols-1 xl:grid-cols-3 gap-4">
      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
        <div class="text-sm opacity-70">Kart Borcu (<?= h1($to) ?> itibariyle)</div>
        <div class="text-2xl font-semibold mt-1"><?= fmt_money($totalCardDebt) ?></div>
        <div class="text-xs opacity-70 mt-2">Açılış borcu + kart giderleri + karttan çıkan transferler − karta ödemeler</div>
      </div>

      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
        <div class="text-sm opacity-70">Yatırım Net Aktarım (<?= h1($to) ?> itibariyle)</div>
        <div class="text-2xl font-semibold mt-1"><?= fmt_money($invNet) ?></div>
        <div class="text-xs opacity-70 mt-2">Yatırıma giden − yatırımdan çıkan (market fiyatı yok, nakit bazlı)</div>
      </div>

      <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
        <div class="text-sm opacity-70">Toplam İşlem (bu rapor)</div>
        <div class="text-2xl font-semibold mt-1"><?= (int)count($rows) ?></div>
        <div class="text-xs opacity-70 mt-2">Liste max 2000 satır</div>
      </div>
    </div>
  <?php endif; ?>

  <?php if ($tab === 'cards'): ?>
    <div class="mt-4 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
      <h3 class="text-lg font-semibold">💳 Kartlar (<?= h1($to) ?> itibariyle)</h3>

      <div class="mt-3 overflow-x-auto">
        <table class="w-full text-sm">
          <thead class="text-xs uppercase opacity-70">
            <tr class="border-b border-slate-200 dark:border-slate-800">
              <th class="text-left py-2 pr-3">Kart</th>
              <th class="text-right py-2 pl-3">Limit</th>
              <th class="text-right py-2 pl-3">Borç</th>
              <th class="text-right py-2 pl-3">Kalan</th>
            </tr>
          </thead>
          <tbody>
            <?php if (!$cardDebtRows): ?>
              <tr><td colspan="4" class="py-8 text-center opacity-70">Kart yok.</td></tr>
            <?php endif; ?>

            <?php foreach ($cardDebtRows as $r): ?>
              <?php
                $opening = (float)$r['opening_debt'];
                $exp = (float)$r['card_expenses'];
                $payments = (float)$r['payments_to_card'];
                $out = (float)$r['card_out_transfers'];
                $debt = $opening + $exp + $out - $payments;
                $limit = (float)$r['card_limit'];
                $remaining = $limit - $debt;
              ?>
              <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
                <td class="py-3 pr-3 font-semibold"><?= h1($r['name']) ?></td>
                <td class="py-3 pl-3 text-right"><?= fmt_money($limit) ?></td>
                <td class="py-3 pl-3 text-right"><?= fmt_money($debt) ?></td>
                <td class="py-3 pl-3 text-right"><?= fmt_money($remaining) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>
  <?php endif; ?>

  <?php if ($tab === 'investments'): ?>
    <div class="mt-4 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
      <h3 class="text-lg font-semibold">📈 Yatırımlar</h3>
      <div class="mt-2 text-sm opacity-80">
        Bu bölüm market fiyatı çekmez. Nakit bazlı net aktarımı gösterir.
      </div>

      <div class="mt-4 grid grid-cols-1 md:grid-cols-3 gap-4">
        <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <div class="text-xs opacity-70">Yatırıma Gelen</div>
          <div class="text-lg font-semibold mt-1"><?= fmt_money($invCash['in']) ?></div>
        </div>
        <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <div class="text-xs opacity-70">Yatırımdan Çıkan</div>
          <div class="text-lg font-semibold mt-1"><?= fmt_money($invCash['out']) ?></div>
        </div>
        <div class="p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <div class="text-xs opacity-70">Net</div>
          <div class="text-lg font-semibold mt-1"><?= fmt_money($invNet) ?></div>
        </div>
      </div>
    </div>
  <?php endif; ?>

  <?php if ($tab === 'goals'): ?>
    <div class="mt-4 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
      <h3 class="text-lg font-semibold">🎯 Hedefler (Bu dönem katkılar)</h3>

      <div class="mt-3 overflow-x-auto">
        <table class="w-full text-sm">
          <thead class="text-xs uppercase opacity-70">
            <tr class="border-b border-slate-200 dark:border-slate-800">
              <th class="text-left py-2 pr-3">Hedef</th>
              <th class="text-right py-2 pl-3">Katkı Toplamı</th>
            </tr>
          </thead>
          <tbody>
            <?php if (!$goalRange): ?>
              <tr><td colspan="2" class="py-8 text-center opacity-70">Bu dönemde hedef katkısı yok.</td></tr>
            <?php endif; ?>
            <?php foreach ($goalRange as $r): ?>
              <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
                <td class="py-3 pr-3 font-semibold"><?= h1($r['name'] ?? '—') ?></td>
                <td class="py-3 pl-3 text-right"><?= fmt_money((float)$r['total']) ?></td>
              </tr>
            <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    </div>
  <?php endif; ?>

  <?php if (in_array($tab, ['income','expense'], true)): ?>
    <div class="mt-4 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
      <h3 class="text-lg font-semibold">
        <?= $tab==='income' ? '💰 Gelir Raporu' : '🧾 Gider Raporu' ?>
      </h3>
      <div class="text-sm opacity-80 mt-1">Filtrelerde “Tür” alanını otomatik ayarlayabilirsin.</div>
    </div>
  <?php endif; ?>

  <!-- Transaction list -->
  <div class="mt-4 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 card">
    <div class="flex items-center justify-between gap-3">
      <h3 class="text-lg font-semibold">📄 İşlem Listesi</h3>
      <div class="text-xs opacity-70">Liste: <?= (int)count($rows) ?> kayıt</div>
    </div>

    <div class="mt-3 overflow-x-auto">
      <table class="w-full text-sm">
        <thead class="text-xs uppercase opacity-70">
          <tr class="border-b border-slate-200 dark:border-slate-800">
            <th class="text-left py-2 pr-3">Tarih</th>
            <th class="text-left py-2 pr-3">Tür</th>
            <th class="text-left py-2 pr-3">Detay</th>
            <th class="text-right py-2 pl-3">Tutar</th>
          </tr>
        </thead>
        <tbody>
          <?php if (!$rows): ?>
            <tr><td colspan="4" class="py-10 text-center opacity-70">Bu filtrelerde kayıt yok.</td></tr>
          <?php endif; ?>

          <?php foreach ($rows as $r): ?>
            <?php
              $detail = build_detail($r, $categories, $accounts, $cards, $investments);
              $kindLabel = $r['kind']==='income' ? 'Gelir' : ($r['kind']==='expense' ? 'Gider' : 'Transfer');
              $amount = (float)$r['amount'];
            ?>
            <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
              <td class="py-3 pr-3"><?= h1($r['occurred_on']) ?></td>
              <td class="py-3 pr-3"><?= h1($kindLabel) ?></td>
              <td class="py-3 pr-3"><?= h1($detail) ?></td>
              <td class="py-3 pl-3 text-right font-semibold"><?= h1(fmt_money_signed((string)$r['kind'], $amount)) ?></td>
            </tr>
          <?php endforeach; ?>
        </tbody>
      </table>
    </div>

    <div class="mt-3 text-xs opacity-70">
      PDF almak için “Yazdır / PDF” butonuna bas → tarayıcıdan “PDF olarak kaydet”.
    </div>
  </div>

</div>

<?php require __DIR__ . '/../inc/footer.php'; ?>
