<?php
declare(strict_types=1);

$type = 'settings';

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';
require __DIR__ . '/../inc/notify.php';
require __DIR__ . '/../inc/backup_lib.php';

if (file_exists(__DIR__ . '/../inc/csrf.php')) require __DIR__ . '/../inc/csrf.php';

$uid = function_exists('require_user_id') ? require_user_id($pdo) : (current_user_id() ?? 0);
if ($uid <= 0) { http_response_code(401); echo "unauthorized"; exit; }

$hasCsrf = function_exists('csrf_token') && function_exists('csrf_verify');

function post_str(string $k, string $def=''): string { return trim((string)($_POST[$k] ?? $def)); }
function post_int(string $k, int $def=0): int { return (int)($_POST[$k] ?? $def); }
function post_bool(string $k): int { return isset($_POST[$k]) ? 1 : 0; }

function ensure_settings_row(PDO $pdo, int $uid): void {
  $st = $pdo->prepare("SELECT user_id FROM settings WHERE user_id=? LIMIT 1");
  $st->execute([$uid]);
  $row = $st->fetch();
  if (!$row) {
    $ins = $pdo->prepare("INSERT INTO settings (user_id) VALUES (?)");
    $ins->execute([$uid]);
  }
}

try {
  ensure_settings_row($pdo, $uid);
} catch (Throwable $e) {
  require __DIR__ . '/../inc/header.php';
  echo '<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-rose-200 dark:border-rose-900">';
  echo '<div class="font-semibold text-rose-700 dark:text-rose-300">Ayarlar tablosu bulunamadı</div>';
  echo '<div class="text-sm opacity-80 mt-2">Önce SQL ile <code>settings</code> tablosunu oluşturmalısın. Hata: '.h($e->getMessage()).'</div>';
  echo '</div>';
  require __DIR__ . '/../inc/footer.php';
  exit;
}

/**
 * cron_token yoksa üret (cron.php bununla güvenli çalışacak)
 * Not: cron_token kolonu yoksa bu blok hata verir; o zaman SQL ile kolon ekle.
 */
try {
  $tokSt = $pdo->prepare("SELECT cron_token FROM settings WHERE user_id=?");
  $tokSt->execute([$uid]);
  $ct = (string)($tokSt->fetchColumn() ?? '');
  if ($ct === '') {
    $newTok = bin2hex(random_bytes(16));
    $u = $pdo->prepare("UPDATE settings SET cron_token=? WHERE user_id=?");
    $u->execute([$newTok, $uid]);
  }
} catch (Throwable $e) {
  // cron_token kolonu yoksa görmezden gel; sayfa yine açılsın.
}

$action = post_str('action','');
$csrf   = post_str('csrf','');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  if ($hasCsrf && !csrf_verify($csrf)) {
    flash_set('error', 'Güvenlik doğrulaması başarısız (CSRF). Sayfayı yenileyip tekrar deneyin.');
    header('Location: ' . page_url('settings.php'));
    exit;
  }

  if ($action === 'save_project') {
    $appName = post_str('app_name');
    if ($appName === '') $appName = 'Kaşmir Gelir Gider Takip';

    $st = $pdo->prepare("UPDATE settings SET app_name=? WHERE user_id=?");
    $st->execute([$appName, $uid]);

    flash_set('success', 'Proje ayarları kaydedildi.');
    header('Location: ' . page_url('settings.php') . '?tab=project');
    exit;
  }

  if ($action === 'save_profile') {
    $full = post_str('full_name');
    $user = post_str('username');
    $mail = post_str('profile_email');

    $st = $pdo->prepare("UPDATE settings SET full_name=?, username=?, profile_email=? WHERE user_id=?");
    $st->execute([$full ?: null, $user ?: null, $mail ?: null, $uid]);

    flash_set('success', 'Profil bilgileri kaydedildi.');
    header('Location: ' . page_url('settings.php') . '?tab=profile');
    exit;
  }

  if ($action === 'save_summary') {
    $dailyT  = post_bool('summary_daily_telegram');
    $dailyE  = post_bool('summary_daily_email');
    $monthT  = post_bool('summary_monthly_telegram');
    $monthE  = post_bool('summary_monthly_email');

    $st = $pdo->prepare("
      UPDATE settings
         SET summary_daily_telegram=?,
             summary_daily_email=?,
             summary_monthly_telegram=?,
             summary_monthly_email=?
       WHERE user_id=?
    ");
    $st->execute([$dailyT,$dailyE,$monthT,$monthE,$uid]);

    flash_set('success', 'Özet tercihleri kaydedildi.');
    header('Location: ' . page_url('settings.php') . '?tab=summary');
    exit;
  }

  if ($action === 'save_api') {
    $bot  = post_str('telegram_bot_token');
    $chat = post_str('telegram_chat_id');

    $st = $pdo->prepare("UPDATE settings SET telegram_bot_token=?, telegram_chat_id=? WHERE user_id=?");
    $st->execute([$bot ?: null, $chat ?: null, $uid]);

    flash_set('success', 'API (Telegram) ayarları kaydedildi.');
    header('Location: ' . page_url('settings.php') . '?tab=api');
    exit;
  }

  if ($action === 'test_telegram') {
    $st = $pdo->prepare("SELECT telegram_bot_token, telegram_chat_id, app_name FROM settings WHERE user_id=?");
    $st->execute([$uid]);
    $ss = $st->fetch(PDO::FETCH_ASSOC) ?: [];

    $txt = "<b>".((string)($ss['app_name'] ?? 'Kaşmir Finance'))."</b>\nTelegram test mesajı ✅\n".date('Y-m-d H:i');
    $r = telegram_send((string)($ss['telegram_bot_token'] ?? ''), (string)($ss['telegram_chat_id'] ?? ''), $txt);

    flash_set($r['ok'] ? 'success' : 'error', $r['ok'] ? 'Telegram testi başarılı.' : ('Telegram testi başarısız: '.$r['error']));
    header('Location: ' . page_url('settings.php') . '?tab=api');
    exit;
  }

  if ($action === 'save_smtp') {
    $host = post_str('smtp_host');
    $user = post_str('smtp_username');
    $pass = post_str('smtp_password');
    $sec  = post_str('smtp_security','ssl');
    $port = post_int('smtp_port', 0);
    $fromEmail = post_str('smtp_from_email');
    $fromName  = post_str('smtp_from_name');

    if (!in_array($sec, ['none','ssl','tls'], true)) $sec = 'ssl';
    $portVal = $port > 0 ? $port : null;

    $st = $pdo->prepare("
      UPDATE settings
         SET smtp_host=?,
             smtp_username=?,
             smtp_password=?,
             smtp_security=?,
             smtp_port=?,
             smtp_from_email=?,
             smtp_from_name=?
       WHERE user_id=?
    ");
    $st->execute([
      $host ?: null,
      $user ?: null,
      $pass ?: null,
      $sec,
      $portVal,
      $fromEmail ?: null,
      $fromName ?: null,
      $uid
    ]);

    flash_set('success', 'SMTP ayarları kaydedildi.');
    header('Location: ' . page_url('settings.php') . '?tab=smtp');
    exit;
  }

  if ($action === 'test_smtp') {
    $st = $pdo->prepare("SELECT * FROM settings WHERE user_id=?");
    $st->execute([$uid]);
    $ss = $st->fetch(PDO::FETCH_ASSOC) ?: [];

    $to = (string)($ss['profile_email'] ?? '');
    if ($to === '') $to = (string)($ss['smtp_from_email'] ?? '');

    $cfg = [
      'host' => (string)($ss['smtp_host'] ?? ''),
      'port' => (int)($ss['smtp_port'] ?? 0),
      'username' => (string)($ss['smtp_username'] ?? ''),
      'password' => (string)($ss['smtp_password'] ?? ''),
      'security' => (string)($ss['smtp_security'] ?? 'ssl'),
      'from_email' => (string)($ss['smtp_from_email'] ?? ''),
      'from_name' => (string)($ss['smtp_from_name'] ?? ((string)($ss['app_name'] ?? 'Kaşmir Finance'))),
    ];

    $html = "<b>SMTP test mesajı ✅</b><br>".date('Y-m-d H:i');
    $r = smtp_send($cfg, $to, 'Kaşmir Finance - SMTP Test', $html);

    flash_set($r['ok'] ? 'success' : 'error', $r['ok'] ? "SMTP testi başarılı. ($to)" : ('SMTP testi başarısız: '.($r['error'] ?? 'unknown')));
    header('Location: ' . page_url('settings.php') . '?tab=smtp');
    exit;
  }

  if ($action === 'save_backup') {
    $enabled = post_bool('backup_enabled');
    $freq = post_str('backup_frequency','weekly');
    $keep = max(1, (int)post_int('backup_keep_count', 10));
    if (!in_array($freq, ['daily','weekly','monthly'], true)) $freq = 'weekly';

    $st = $pdo->prepare("
      UPDATE settings
         SET backup_enabled=?,
             backup_frequency=?,
             backup_keep_count=?
       WHERE user_id=?
    ");
    $st->execute([$enabled,$freq,$keep,$uid]);

    flash_set('success', 'Yedekleme ayarları kaydedildi.');
    header('Location: ' . page_url('settings.php') . '?tab=backup');
    exit;
  }

  if ($action === 'backup_now') {
    $r = backup_create($pdo);
    flash_set($r['ok'] ? 'success' : 'error', $r['ok'] ? ('Yedek alındı: '.$r['file']) : ('Yedek alınamadı: '.$r['error']));
    header('Location: ' . page_url('settings.php') . '?tab=backup');
    exit;
  }

  flash_set('error', 'Bilinmeyen işlem.');
  header('Location: ' . page_url('settings.php'));
  exit;
}

// Read settings row
$st = $pdo->prepare("SELECT * FROM settings WHERE user_id=? LIMIT 1");
$st->execute([$uid]);
$s = $st->fetch(PDO::FETCH_ASSOC) ?: [];

$tab = trim((string)($_GET['tab'] ?? 'project'));
$tabs = [
  'project' => '🧩 Proje',
  'profile' => '👤 Profil',
  'summary' => '📌 Özet Tercihleri',
  'api'     => '🔌 API',
  'smtp'    => '✉️ SMTP',
  'backup'  => '🗄️ Yedekleme',
];
if (!isset($tabs[$tab])) $tab = 'project';

require __DIR__ . '/../inc/header.php';
?>

<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
  <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
    <div>
      <h2 class="text-lg font-semibold">⚙️ Ayarlar</h2>
      <p class="text-sm opacity-70">Proje adı, profil, özetler, Telegram / SMTP ve yedekleme seçenekleri.</p>
    </div>
  </div>

  <div class="mt-4 flex flex-wrap gap-2">
    <?php foreach ($tabs as $k=>$label): ?>
      <a href="<?= h(page_url('settings.php')) ?>?tab=<?= h($k) ?>"
         class="px-3 py-2 rounded-xl text-sm border
                <?= $tab===$k
                  ? 'bg-slate-900 text-white border-slate-900 dark:bg-white dark:text-slate-900 dark:border-white'
                  : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>">
        <?= $label ?>
      </a>
    <?php endforeach; ?>
  </div>
</div>

<?= function_exists('flash_render') ? flash_render() : '' ?>

<div class="mt-4 grid grid-cols-1 xl:grid-cols-3 gap-4">

  <div class="xl:col-span-2 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">

    <?php if ($tab === 'project'): ?>
      <h3 class="text-lg font-semibold">🧩 Proje Ayarları</h3>
      <p class="text-sm opacity-70 mt-1">Uygulama adı (başlıklarda ve menüde kullanılabilir).</p>

      <form method="post" class="mt-4 space-y-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="save_project">

        <div>
          <label class="text-sm opacity-70">Uygulama Adı</label>
          <input name="app_name" value="<?= h((string)($s['app_name'] ?? 'Kaşmir Gelir Gider Takip')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="Örn: Kaşmir Finance">
        </div>

        <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Kaydet</button>
      </form>

    <?php elseif ($tab === 'profile'): ?>
      <h3 class="text-lg font-semibold">👤 Profil Bilgileri</h3>
      <p class="text-sm opacity-70 mt-1">İsim, kullanıcı adı ve e-posta bilgileri.</p>

      <form method="post" class="mt-4 space-y-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="save_profile">

        <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label class="text-sm opacity-70">İsim Soyisim</label>
            <input name="full_name" value="<?= h((string)($s['full_name'] ?? '')) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          </div>
          <div>
            <label class="text-sm opacity-70">Kullanıcı Adı</label>
            <input name="username" value="<?= h((string)($s['username'] ?? '')) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          </div>
        </div>

        <div>
          <label class="text-sm opacity-70">E-posta</label>
          <input type="email" name="profile_email" value="<?= h((string)($s['profile_email'] ?? '')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Kaydet</button>
      </form>

    <?php elseif ($tab === 'summary'): ?>
      <h3 class="text-lg font-semibold">📌 Özet Tercihleri</h3>
      <p class="text-sm opacity-70 mt-1">Günlük/Aylık özetlerin hangi kanala gideceğini seç.</p>

      <form method="post" class="mt-4 space-y-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="save_summary">

        <?php
          $dT = (int)($s['summary_daily_telegram'] ?? 0);
          $dE = (int)($s['summary_daily_email'] ?? 0);
          $mT = (int)($s['summary_monthly_telegram'] ?? 0);
          $mE = (int)($s['summary_monthly_email'] ?? 0);
        ?>

        <div class="space-y-2 text-sm">
          <?php
            $items = [
              ['summary_daily_telegram', 'Günlük Telegram Özeti', $dT],
              ['summary_daily_email', 'Günlük E-posta Özeti', $dE],
              ['summary_monthly_telegram', 'Aylık Telegram Özeti', $mT],
              ['summary_monthly_email', 'Aylık E-posta Özeti', $mE],
            ];
            foreach ($items as [$name,$label,$val]):
          ?>
            <label class="flex items-center gap-3 p-3 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
              <input type="checkbox" name="<?= h($name) ?>" <?= $val ? 'checked' : '' ?>>
              <span class="font-semibold"><?= h($label) ?></span>
              <span class="ml-auto text-xs opacity-70"><?= str_contains($name,'telegram') ? 'Telegram' : 'E-posta' ?></span>
            </label>
          <?php endforeach; ?>
        </div>

        <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Kaydet</button>
      </form>

      <div class="mt-4 p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 text-sm">
        <div class="font-semibold">Cron / Otomatik Gönderim</div>
        <div class="opacity-80 mt-2">
          Cron URL (tarayıcıdan da çalışır):
          <div class="mt-2 font-mono text-xs p-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
            <?= h(page_url('cron.php')) ?>?uid=<?= (int)$uid ?>&token=<?= h((string)($s['cron_token'] ?? '')) ?>
          </div>
        </div>
      </div>

    <?php elseif ($tab === 'api'): ?>
      <h3 class="text-lg font-semibold">🔌 API Ayarları</h3>
      <p class="text-sm opacity-70 mt-1">Telegram (ileride SMS/diğer entegrasyonlar buraya eklenebilir).</p>

      <form method="post" class="mt-4 space-y-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="save_api">

        <div>
          <label class="text-sm opacity-70">Telegram Bot Token</label>
          <input name="telegram_bot_token" value="<?= h((string)($s['telegram_bot_token'] ?? '')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="123456:ABC-DEF...">
        </div>

        <div>
          <label class="text-sm opacity-70">Telegram Chat ID</label>
          <input name="telegram_chat_id" value="<?= h((string)($s['telegram_chat_id'] ?? '')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="Örn: 123456789">
        </div>

        <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Kaydet</button>
      </form>

      <form method="post" class="mt-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="test_telegram">
        <button class="px-4 py-2 rounded-xl border border-slate-200 dark:border-slate-800">📨 Test Telegram</button>
      </form>

    <?php elseif ($tab === 'smtp'): ?>
      <h3 class="text-lg font-semibold">✉️ SMTP Ayarları</h3>
      <p class="text-sm opacity-70 mt-1">E-posta özetleri için SMTP bilgilerini gir.</p>

      <form method="post" class="mt-4 space-y-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="save_smtp">

        <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label class="text-sm opacity-70">SMTP Host</label>
            <input name="smtp_host" value="<?= h((string)($s['smtp_host'] ?? '')) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   placeholder="mail.domain.com">
          </div>
          <div>
            <label class="text-sm opacity-70">Port</label>
            <input type="number" name="smtp_port" value="<?= h((string)($s['smtp_port'] ?? '')) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   placeholder="465 / 587">
          </div>
        </div>

        <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label class="text-sm opacity-70">Kullanıcı Adı</label>
            <input name="smtp_username" value="<?= h((string)($s['smtp_username'] ?? '')) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   placeholder="v3@domain.com">
          </div>
          <div>
            <label class="text-sm opacity-70">Şifre</label>
            <input type="password" name="smtp_password" value="<?= h((string)($s['smtp_password'] ?? '')) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   placeholder="••••••••">
          </div>
        </div>

        <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label class="text-sm opacity-70">Güvenlik</label>
            <?php $sec = (string)($s['smtp_security'] ?? 'ssl'); ?>
            <select name="smtp_security"
                    class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
              <option value="ssl" <?= $sec==='ssl'?'selected':'' ?>>SSL</option>
              <option value="tls" <?= $sec==='tls'?'selected':'' ?>>TLS</option>
              <option value="none" <?= $sec==='none'?'selected':'' ?>>Yok</option>
            </select>
          </div>
          <div>
            <label class="text-sm opacity-70">Gönderen E-posta</label>
            <input name="smtp_from_email" value="<?= h((string)($s['smtp_from_email'] ?? '')) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   placeholder="v3@domain.com">
          </div>
        </div>

        <div>
          <label class="text-sm opacity-70">Gönderen Adı</label>
          <input name="smtp_from_name" value="<?= h((string)($s['smtp_from_name'] ?? '')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="Kaşmir Finance">
        </div>

        <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Kaydet</button>
      </form>

      <form method="post" class="mt-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="test_smtp">
        <button class="px-4 py-2 rounded-xl border border-slate-200 dark:border-slate-800">✉️ Test SMTP</button>
      </form>

    <?php elseif ($tab === 'backup'): ?>
      <h3 class="text-lg font-semibold">🗄️ Yedekleme</h3>
      <p class="text-sm opacity-70 mt-1">Şimdi yedek al, listele ve indir.</p>

      <form method="post" class="mt-4 space-y-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="save_backup">

        <?php $be=(int)($s['backup_enabled'] ?? 0); $bf=(string)($s['backup_frequency'] ?? 'weekly'); $bk=(int)($s['backup_keep_count'] ?? 10); ?>

        <label class="flex items-center gap-3 p-3 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <input type="checkbox" name="backup_enabled" <?= $be ? 'checked' : '' ?>>
          <span class="font-semibold">Yedekleme Aktif</span>
        </label>

        <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
          <div>
            <label class="text-sm opacity-70">Sıklık</label>
            <select name="backup_frequency"
                    class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
              <option value="daily" <?= $bf==='daily'?'selected':'' ?>>Günlük</option>
              <option value="weekly" <?= $bf==='weekly'?'selected':'' ?>>Haftalık</option>
              <option value="monthly" <?= $bf==='monthly'?'selected':'' ?>>Aylık</option>
            </select>
          </div>
          <div>
            <label class="text-sm opacity-70">Kaç yedek saklansın?</label>
            <input type="number" name="backup_keep_count" value="<?= h((string)$bk) ?>"
                   class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                   min="1" max="365">
          </div>
        </div>

        <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">Kaydet</button>
      </form>

      <form method="post" class="mt-3">
        <?php if ($hasCsrf): ?><input type="hidden" name="csrf" value="<?= h(csrf_token()) ?>"><?php endif; ?>
        <input type="hidden" name="action" value="backup_now">
        <button class="px-4 py-2 rounded-xl border border-slate-200 dark:border-slate-800">🗄️ Şimdi Yedek Al</button>
      </form>

      <?php $backs = function_exists('backup_list') ? backup_list(30) : []; ?>
      <div class="mt-4">
        <h4 class="font-semibold">Son Yedekler</h4>
        <div class="mt-2 space-y-2 text-sm">
          <?php if (!$backs): ?>
            <div class="opacity-70">Henüz yedek yok.</div>
          <?php endif; ?>
          <?php foreach ($backs as $b): ?>
            <div class="p-3 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 flex items-center gap-3">
              <div class="font-mono"><?= h((string)$b['name']) ?></div>
              <div class="ml-auto opacity-70">
                <?= number_format(((float)$b['size'])/1024, 1, ',', '.') ?> KB • <?= date('Y-m-d H:i', (int)$b['mtime']) ?>
              </div>
              <a class="underline" href="<?= h(page_url('download_backup.php')) ?>?f=<?= urlencode((string)$b['name']) ?>">İndir</a>
            </div>
          <?php endforeach; ?>
        </div>
      </div>

    <?php endif; ?>
  </div>

  <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h3 class="text-lg font-semibold">🧾 Hızlı Bilgi</h3>

    <div class="mt-3 space-y-2 text-sm">
      <div class="flex justify-between gap-3">
        <span class="opacity-70">Uygulama Adı</span>
        <span class="font-semibold"><?= h((string)($s['app_name'] ?? 'Kaşmir Gelir Gider Takip')) ?></span>
      </div>
      <div class="flex justify-between gap-3">
        <span class="opacity-70">Telegram</span>
        <span class="font-semibold"><?= !empty($s['telegram_bot_token']) && !empty($s['telegram_chat_id']) ? 'Bağlı' : '—' ?></span>
      </div>
      <div class="flex justify-between gap-3">
        <span class="opacity-70">SMTP</span>
        <span class="font-semibold"><?= !empty($s['smtp_host']) && !empty($s['smtp_username']) ? 'Hazır' : '—' ?></span>
      </div>
      <div class="flex justify-between gap-3">
        <span class="opacity-70">Cron Token</span>
        <span class="font-mono text-xs"><?= h((string)($s['cron_token'] ?? '')) ?></span>
      </div>
    </div>

    <div class="mt-4 p-4 rounded-2xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800 text-xs">
      Cron URL:
      <div class="mt-2 font-mono">
        <?= h(page_url('cron.php')) ?>?uid=<?= (int)$uid ?>&token=<?= h((string)($s['cron_token'] ?? '')) ?>
      </div>
    </div>
  </div>

</div>

<?php require __DIR__ . '/../inc/footer.php'; ?>
