<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';

$uid = function_exists('require_user_id') ? require_user_id($pdo) : (current_user_id() ?? 0);
if ($uid <= 0) {
  if (function_exists('flash_set')) flash_set('error', 'Users tablosunda kullanıcı bulunamadı. Lütfen 1 kullanıcı ekle.');
  http_response_code(500);
  echo "Users tablosunda kullanıcı yok.";
  exit;
}

function post_str(string $k): string { return trim((string)($_POST[$k] ?? '')); }
function post_float(string $k): float { return (float)($_POST[$k] ?? 0); }
function post_int(string $k): int { return (int)($_POST[$k] ?? 0); }

function money_fmt(float $v): string {
  return number_format($v, 2, ',', '.');
}

function valid_currency(string $c): string {
  $c = strtoupper(trim($c));
  return in_array($c, ['TRY','USD','EUR'], true) ? $c : 'TRY';
}

function insert_transfer(PDO $pdo, int $uid, string $date, float $amount, string $cur,
                         string $sourceType, ?int $sourceId,
                         string $destType, ?int $destId,
                         ?int $categoryId, string $desc): void {
  $st = $pdo->prepare("
    INSERT INTO transactions
      (user_id, kind, occurred_on, amount, currency, category_id, description, source_type, source_id, dest_type, dest_id)
    VALUES
      (?, 'transfer', ?, ?, ?, ?, ?, ?, ?, ?, ?)
  ");
  $st->execute([
    $uid, $date, $amount, $cur,
    $categoryId ?: null,
    $desc !== '' ? $desc : null,
    $sourceType, $sourceId,
    $destType, $destId
  ]);
}

function insert_expense(PDO $pdo, int $uid, string $date, float $amount, string $cur,
                        ?int $categoryId, string $sourceType, ?int $sourceId,
                        string $desc): void {
  $st = $pdo->prepare("
    INSERT INTO transactions
      (user_id, kind, occurred_on, amount, currency, category_id, description, source_type, source_id, dest_type, dest_id)
    VALUES
      (?, 'expense', ?, ?, ?, ?, ?, ?, ?, 'none', NULL)
  ");
  $st->execute([
    $uid, $date, $amount, $cur,
    $categoryId ?: null,
    $desc !== '' ? $desc : null,
    $sourceType, $sourceId
  ]);
}

/* -------------------------------------------------------
   Load sources
------------------------------------------------------- */
$accounts = $pdo->prepare("SELECT id,name,currency FROM accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$accounts->execute([$uid]);
$accounts = $accounts->fetchAll();

$cards = $pdo->prepare("SELECT id,name,currency FROM credit_cards WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$cards->execute([$uid]);
$cards = $cards->fetchAll();

$inv = $pdo->prepare("SELECT id,name,currency FROM investment_accounts WHERE user_id=? AND deleted_at IS NULL AND is_active=1 ORDER BY name");
$inv->execute([$uid]);
$inv = $inv->fetchAll();

/* Optional: expense categories for "cash advance fee" if you want */
$expenseCats = $pdo->prepare("SELECT id,name FROM categories WHERE user_id=? AND type='expense' AND deleted_at IS NULL AND is_active=1 ORDER BY sort_order,name");
$expenseCats->execute([$uid]);
$expenseCats = $expenseCats->fetchAll();

/* -------------------------------------------------------
   Handle actions
------------------------------------------------------- */
$action = (string)($_POST['action'] ?? '');
if ($action) {
  $date = post_str('date');
  if ($date === '') $date = date('Y-m-d');

  $amount = post_float('amount');
  $cur = valid_currency(post_str('currency') ?: 'TRY');
  $note = post_str('note');

  try {
    if ($amount <= 0) throw new RuntimeException('Tutar 0’dan büyük olmalı.');

    if ($action === 'acc_to_acc') {
      $from = post_int('from_account_id');
      $to   = post_int('to_account_id');
      if ($from <= 0 || $to <= 0) throw new RuntimeException('Gönderici ve alıcı hesap seç.');
      if ($from === $to) throw new RuntimeException('Aynı hesaba transfer yapılamaz.');
      insert_transfer($pdo, $uid, $date, $amount, $cur, 'account', $from, 'account', $to, null, $note ?: 'Hesaplar arası transfer');
      flash_set('success', 'Transfer kaydedildi ✅');
      header('Location: '.page_url('transfers.php').'?tab=acc');
      exit;
    }

    if ($action === 'acc_to_card') {
      $from = post_int('from_account_id');
      $to   = post_int('to_card_id');
      if ($from <= 0 || $to <= 0) throw new RuntimeException('Hesap ve kredi kartı seç.');
      insert_transfer($pdo, $uid, $date, $amount, $cur, 'account', $from, 'credit_card', $to, null, $note ?: 'Kredi kartı ödemesi');
      flash_set('success', 'Kart ödemesi kaydedildi ✅');
      header('Location: '.page_url('transfers.php').'?tab=card');
      exit;
    }

    if ($action === 'card_to_acc') {
      $from = post_int('from_card_id');
      $to   = post_int('to_account_id');
      if ($from <= 0 || $to <= 0) throw new RuntimeException('Kredi kartı ve hedef hesap seç.');
      insert_transfer($pdo, $uid, $date, $amount, $cur, 'credit_card', $from, 'account', $to, null, $note ?: 'Karttan nakit çekim/transfer');
      flash_set('success', 'Kart → Hesap transferi kaydedildi ✅');
      header('Location: '.page_url('transfers.php').'?tab=card');
      exit;
    }

    if ($action === 'cash_advance') {
      $cardId   = post_int('from_card_id');
      $accId    = post_int('to_account_id');
      $feeType  = post_str('fee_type'); // percent|fixed|none
      $feeVal   = post_float('fee_value');
      $feeCatId = post_int('fee_category_id') ?: null;

      if ($cardId <= 0 || $accId <= 0) throw new RuntimeException('Kredi kartı ve hedef hesap seç.');
      if ($feeType !== 'none' && $feeVal < 0) throw new RuntimeException('Faiz/masraf negatif olamaz.');

      $pdo->beginTransaction();

      // principal transfer
      insert_transfer($pdo, $uid, $date, $amount, $cur, 'credit_card', $cardId, 'account', $accId, null, $note ?: 'Nakit avans');

      // optional fee as expense on credit card
      $feeAmount = 0.0;
      if ($feeType === 'percent') {
        $feeAmount = $amount * ($feeVal / 100.0);
      } elseif ($feeType === 'fixed') {
        $feeAmount = $feeVal;
      }

      if ($feeType !== 'none' && $feeAmount > 0) {
        insert_expense($pdo, $uid, $date, $feeAmount, $cur, $feeCatId, 'credit_card', $cardId, 'Nakit avans faiz/masraf');
      }

      $pdo->commit();

      $msg = 'Nakit avans kaydedildi ✅';
      if ($feeAmount > 0) $msg .= ' (Masraf: '.money_fmt($feeAmount).' '.$cur.')';
      flash_set('success', $msg);

      header('Location: '.page_url('transfers.php').'?tab=advance');
      exit;
    }

    if ($action === 'acc_to_inv') {
      $from = post_int('from_account_id');
      $to   = post_int('to_inv_id');
      if ($from <= 0 || $to <= 0) throw new RuntimeException('Hesap ve yatırım hesabı seç.');
      insert_transfer($pdo, $uid, $date, $amount, $cur, 'account', $from, 'investment', $to, null, $note ?: 'Yatırıma para gönder');
      flash_set('success', 'Yatırıma para gönderme kaydedildi ✅');
      header('Location: '.page_url('transfers.php').'?tab=inv');
      exit;
    }

    if ($action === 'inv_to_acc') {
      $from = post_int('from_inv_id');
      $to   = post_int('to_account_id');
      if ($from <= 0 || $to <= 0) throw new RuntimeException('Yatırım hesabı ve hedef hesap seç.');
      insert_transfer($pdo, $uid, $date, $amount, $cur, 'investment', $from, 'account', $to, null, $note ?: 'Yatırımdan para çek');
      flash_set('success', 'Yatırımdan para çekme kaydedildi ✅');
      header('Location: '.page_url('transfers.php').'?tab=inv');
      exit;
    }

    if ($action === 'inv_to_inv') {
      $from = post_int('from_inv_id');
      $to   = post_int('to_inv_id');
      if ($from <= 0 || $to <= 0) throw new RuntimeException('Gönderici ve alıcı yatırım hesabı seç.');
      if ($from === $to) throw new RuntimeException('Aynı yatırım hesabına transfer yapılamaz.');
      insert_transfer($pdo, $uid, $date, $amount, $cur, 'investment', $from, 'investment', $to, null, $note ?: 'Yatırımlar arası transfer');
      flash_set('success', 'Yatırım transferi kaydedildi ✅');
      header('Location: '.page_url('transfers.php').'?tab=inv');
      exit;
    }

    throw new RuntimeException('Geçersiz işlem.');
  } catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    flash_set('error', $e->getMessage());
    $fallbackTab = (string)($_GET['tab'] ?? 'acc');
    header('Location: '.page_url('transfers.php').'?tab='.$fallbackTab);
    exit;
  }
}

$tab = (string)($_GET['tab'] ?? 'acc');
$tab = in_array($tab, ['acc','card','advance','inv'], true) ? $tab : 'acc';

require __DIR__ . '/../inc/header.php';
?>

<div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
  <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
    <div>
      <h2 class="text-lg font-semibold">🔁 Transferler</h2>
      <p class="text-sm opacity-70">Hesap ↔ Kart ↔ Yatırım arası para transferleri</p>
    </div>
    <a href="<?= h(page_url('trash.php')) ?>?entity=transactions"
       class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
      🗑️ Çöp Kutusu
    </a>
  </div>
</div>

<div class="mt-4 flex flex-wrap gap-2">
  <a href="<?= h(page_url('transfers.php')) ?>?tab=acc"
     class="px-4 py-2 rounded-xl border <?= $tab==='acc' ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>">
    🏦 Hesap ↔ Hesap
  </a>
  <a href="<?= h(page_url('transfers.php')) ?>?tab=card"
     class="px-4 py-2 rounded-xl border <?= $tab==='card' ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>">
    💳 Hesap ↔ Kart
  </a>
  <a href="<?= h(page_url('transfers.php')) ?>?tab=advance"
     class="px-4 py-2 rounded-xl border <?= $tab==='advance' ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>">
    💵 Nakit Avans
  </a>
  <a href="<?= h(page_url('transfers.php')) ?>?tab=inv"
     class="px-4 py-2 rounded-xl border <?= $tab==='inv' ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800' ?>">
    📈 Yatırımlar
  </a>
</div>

<?php if ($tab === 'acc'): ?>
  <div class="mt-4 grid grid-cols-1 xl:grid-cols-2 gap-4">
    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">🏦 Hesaplar Arası Transfer</h3>

      <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
        <input type="hidden" name="action" value="acc_to_acc">

        <div>
          <label class="text-sm opacity-70">Gönderici Hesap</label>
          <select name="from_account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($accounts as $a): ?>
              <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Hedef Hesap</label>
          <select name="to_account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($accounts as $a): ?>
              <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tutar</label>
          <input name="amount" type="number" step="0.01" required
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div>
          <label class="text-sm opacity-70">Para Birimi</label>
          <select name="currency" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="TRY">TRY</option>
            <option value="USD">USD</option>
            <option value="EUR">EUR</option>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tarih</label>
          <input name="date" type="date" value="<?= h(date('Y-m-d')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Not (ops.)</label>
          <input name="note"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="Opsiyonel açıklama">
        </div>

        <div class="md:col-span-2 pt-2">
          <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Transferi Kaydet</button>
        </div>
      </form>
    </div>

    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">📌 İpucu</h3>
      <p class="text-sm opacity-70 mt-2">
        Bu sayfada kaydedilen her transfer, <code>transactions</code> tablosuna <b>kind=transfer</b> olarak yazılır.
        Silersen çöp kutusuna gider.
      </p>
    </div>
  </div>
<?php endif; ?>

<?php if ($tab === 'card'): ?>
  <div class="mt-4 grid grid-cols-1 xl:grid-cols-2 gap-4">

    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">🏦 Hesap → 💳 Kredi Kartı (Ödeme)</h3>

      <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
        <input type="hidden" name="action" value="acc_to_card">

        <div>
          <label class="text-sm opacity-70">Nakit/Hesap</label>
          <select name="from_account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($accounts as $a): ?>
              <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Hedef Kredi Kartı</label>
          <select name="to_card_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($cards as $c): ?>
              <option value="<?= (int)$c['id'] ?>"><?= h($c['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tutar</label>
          <input name="amount" type="number" step="0.01" required
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div>
          <label class="text-sm opacity-70">Para Birimi</label>
          <select name="currency" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="TRY">TRY</option>
            <option value="USD">USD</option>
            <option value="EUR">EUR</option>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tarih</label>
          <input name="date" type="date" value="<?= h(date('Y-m-d')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Not (ops.)</label>
          <input name="note"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="Örn: Ocak ekstresi ödeme">
        </div>

        <div class="md:col-span-2 pt-2">
          <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Ödemeyi Kaydet</button>
        </div>
      </form>
    </div>

    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">💳 Kredi Kartı → 🏦 Hesap</h3>

      <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
        <input type="hidden" name="action" value="card_to_acc">

        <div>
          <label class="text-sm opacity-70">Kredi Kartı</label>
          <select name="from_card_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($cards as $c): ?>
              <option value="<?= (int)$c['id'] ?>"><?= h($c['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Hedef Hesap</label>
          <select name="to_account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($accounts as $a): ?>
              <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tutar</label>
          <input name="amount" type="number" step="0.01" required
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div>
          <label class="text-sm opacity-70">Para Birimi</label>
          <select name="currency" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="TRY">TRY</option>
            <option value="USD">USD</option>
            <option value="EUR">EUR</option>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tarih</label>
          <input name="date" type="date" value="<?= h(date('Y-m-d')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Not (ops.)</label>
          <input name="note" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="Örn: Karttan nakde">
        </div>

        <div class="md:col-span-2 pt-2">
          <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Transferi Kaydet</button>
        </div>
      </form>
    </div>

  </div>
<?php endif; ?>

<?php if ($tab === 'advance'): ?>
  <div class="mt-4 grid grid-cols-1 xl:grid-cols-2 gap-4">
    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">💵 Nakit Avans</h3>
      <p class="text-sm opacity-70 mt-1">Karttan nakit çekim + opsiyonel faiz/masraf (ek gider kaydı)</p>

      <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
        <input type="hidden" name="action" value="cash_advance">

        <div>
          <label class="text-sm opacity-70">Kredi Kartı</label>
          <select name="from_card_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($cards as $c): ?>
              <option value="<?= (int)$c['id'] ?>"><?= h($c['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Hedef Hesap (Nakit)</label>
          <select name="to_account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($accounts as $a): ?>
              <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Anapara Tutar</label>
          <input name="amount" type="number" step="0.01" required
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div>
          <label class="text-sm opacity-70">Para Birimi</label>
          <select name="currency" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="TRY">TRY</option>
            <option value="USD">USD</option>
            <option value="EUR">EUR</option>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tarih</label>
          <input name="date" type="date" value="<?= h(date('Y-m-d')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Not (ops.)</label>
          <input name="note" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="Örn: ATM nakit avans">
        </div>

        <div>
          <label class="text-sm opacity-70">Masraf Tipi</label>
          <select name="fee_type" id="fee_type"
                  class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="none">Yok</option>
            <option value="percent">Yüzde (%)</option>
            <option value="fixed">Sabit Tutar</option>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Masraf Değeri</label>
          <input name="fee_value" id="fee_value" type="number" step="0.01" value="0"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
          <p class="text-xs opacity-60 mt-1">Yüzde seçiliyse: anaparanın %’si. Sabitse: doğrudan tutar.</p>
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Masraf Kategorisi (ops.)</label>
          <select name="fee_category_id"
                  class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="">Seçmesen de olur</option>
            <?php foreach ($expenseCats as $ec): ?>
              <option value="<?= (int)$ec['id'] ?>"><?= h($ec['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div class="md:col-span-2 pt-2">
          <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Nakit Avansı Kaydet</button>
        </div>
      </form>
    </div>

    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">📌 Nasıl kaydediyor?</h3>
      <ul class="mt-3 text-sm opacity-80 space-y-2 list-disc pl-5">
        <li>Anapara: <b>transfer</b> (source=credit_card → dest=account)</li>
        <li>Masraf/faiz: seçilirse <b>expense</b> (source=credit_card)</li>
        <li>Bu sayede kart borcu artışı doğru yansır.</li>
      </ul>
    </div>
  </div>
<?php endif; ?>

<?php if ($tab === 'inv'): ?>
  <div class="mt-4 grid grid-cols-1 xl:grid-cols-2 gap-4">

    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">🏦 Hesap → 📈 Yatırım</h3>

      <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
        <input type="hidden" name="action" value="acc_to_inv">

        <div>
          <label class="text-sm opacity-70">Nakit/Hesap</label>
          <select name="from_account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($accounts as $a): ?>
              <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Hedef Yatırım Hesabı</label>
          <select name="to_inv_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($inv as $x): ?>
              <option value="<?= (int)$x['id'] ?>"><?= h($x['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tutar</label>
          <input name="amount" type="number" step="0.01" required
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div>
          <label class="text-sm opacity-70">Para Birimi</label>
          <select name="currency" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="TRY">TRY</option>
            <option value="USD">USD</option>
            <option value="EUR">EUR</option>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tarih</label>
          <input name="date" type="date" value="<?= h(date('Y-m-d')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Not (ops.)</label>
          <input name="note"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800"
                 placeholder="Örn: Binance’e para gönder">
        </div>

        <div class="md:col-span-2 pt-2">
          <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Transferi Kaydet</button>
        </div>
      </form>
    </div>

    <div class="p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
      <h3 class="text-lg font-semibold">📈 Yatırım → 🏦 Hesap / 📈 Yatırım</h3>

      <form class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-3" method="post">
        <div>
          <label class="text-sm opacity-70">Gönderici Yatırım</label>
          <select name="from_inv_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800" required>
            <option value="">Seç...</option>
            <?php foreach ($inv as $x): ?>
              <option value="<?= (int)$x['id'] ?>"><?= h($x['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Hedef Tip</label>
          <select id="inv_dest_type" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="account">Hesap</option>
            <option value="investment">Yatırım</option>
          </select>
        </div>

        <div id="inv_dest_account_wrap">
          <label class="text-sm opacity-70">Hedef Hesap</label>
          <select name="to_account_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="">Seç...</option>
            <?php foreach ($accounts as $a): ?>
              <option value="<?= (int)$a['id'] ?>"><?= h($a['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div id="inv_dest_inv_wrap" class="hidden">
          <label class="text-sm opacity-70">Hedef Yatırım</label>
          <select name="to_inv_id" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="">Seç...</option>
            <?php foreach ($inv as $x): ?>
              <option value="<?= (int)$x['id'] ?>"><?= h($x['name']) ?></option>
            <?php endforeach; ?>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tutar</label>
          <input name="amount" type="number" step="0.01" required
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div>
          <label class="text-sm opacity-70">Para Birimi</label>
          <select name="currency" class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
            <option value="TRY">TRY</option>
            <option value="USD">USD</option>
            <option value="EUR">EUR</option>
          </select>
        </div>

        <div>
          <label class="text-sm opacity-70">Tarih</label>
          <input name="date" type="date" value="<?= h(date('Y-m-d')) ?>"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <div class="md:col-span-2">
          <label class="text-sm opacity-70">Not (ops.)</label>
          <input name="note"
                 class="mt-1 w-full px-3 py-2 rounded-xl bg-slate-50 dark:bg-slate-900 border border-slate-200 dark:border-slate-800">
        </div>

        <input type="hidden" id="inv_action" name="action" value="inv_to_acc">

        <div class="md:col-span-2 pt-2">
          <button class="px-4 py-2 rounded-xl bg-primary text-white font-semibold">✅ Transferi Kaydet</button>
        </div>
      </form>
    </div>

  </div>

  <script>
    const destType = document.getElementById('inv_dest_type');
    const accWrap  = document.getElementById('inv_dest_account_wrap');
    const invWrap  = document.getElementById('inv_dest_inv_wrap');
    const actionEl = document.getElementById('inv_action');

    function syncInvDest(){
      const v = destType.value;
      if (v === 'investment'){
        accWrap.classList.add('hidden');
        invWrap.classList.remove('hidden');
        actionEl.value = 'inv_to_inv';
      } else {
        invWrap.classList.add('hidden');
        accWrap.classList.remove('hidden');
        actionEl.value = 'inv_to_acc';
      }
    }
    destType.addEventListener('change', syncInvDest);
    syncInvDest();
  </script>
<?php endif; ?>

<?php require __DIR__ . '/../inc/footer.php'; ?>
