<?php
declare(strict_types=1);

require __DIR__ . '/../inc/db.php';
require __DIR__ . '/../inc/auth.php';
require __DIR__ . '/../inc/functions.php';

$uid = current_user_id() ?? 1;

$ENTITY = [
  'transactions' => [
    'title' => '🧾 İşlemler',
    'table' => 'transactions',
    'cols'  => ['id','occurred_on','kind','amount','currency','description'],
    'label' => function(array $r): string {
      $kind = $r['kind'] === 'income' ? 'Gelir' : ($r['kind'] === 'expense' ? 'Gider' : 'Transfer');
      $cur = $r['currency'] === 'TRY' ? '₺' : ($r['currency'] === 'USD' ? '$' : '€');
      return "{$r['occurred_on']} • {$kind} • {$cur}".number_format((float)$r['amount'],2,',','.');
    },
    'order' => 'deleted_at DESC, id DESC',
  ],

  'accounts' => [
    'title' => '🏦 Hesaplar',
    'table' => 'accounts',
    'cols'  => ['id','name','currency','opening_balance'],
    'label' => fn($r) => "{$r['name']} • {$r['currency']} • Açılış: ".number_format((float)$r['opening_balance'],2,',','.'),
    'order' => 'deleted_at DESC, id DESC',
  ],

  'credit_cards' => [
    'title' => '💳 Kartlar',
    'table' => 'credit_cards',
    'cols'  => ['id','name','currency','card_limit','opening_debt'],
    'label' => fn($r) => "{$r['name']} • {$r['currency']} • Limit: ".number_format((float)$r['card_limit'],2,',','.')." • Borç: ".number_format((float)$r['opening_debt'],2,',','.'),
    'order' => 'deleted_at DESC, id DESC',
  ],

  'categories' => [
    'title' => '🏷️ Kategoriler',
    'table' => 'categories',
    'cols'  => ['id','type','name'],
    'label' => fn($r) => strtoupper((string)$r['type'])." • ".$r['name'],
    'order' => 'deleted_at DESC, id DESC',
  ],

  'goals' => [
    'title' => '🎯 Hedefler',
    'table' => 'goals',
    'cols'  => ['id','name','target_amount','target_date','status'],
    'label' => fn($r) => "{$r['name']} • Hedef: ".number_format((float)$r['target_amount'],2,',','.')." • {$r['target_date']} • {$r['status']}",
    'order' => 'deleted_at DESC, id DESC',
  ],

  'goal_contributions' => [
    'title' => '💰 Hedef Katkıları',
    'table' => 'goal_contributions',
    'cols'  => ['id','goal_id','account_id','amount','contributed_at','note'],
    'label' => fn($r) => "{$r['contributed_at']} • Tutar: ".number_format((float)$r['amount'],2,',','.').($r['note'] ? " • {$r['note']}" : ''),
    'order' => 'deleted_at DESC, id DESC',
  ],

  'investment_accounts' => [
    'title' => '📈 Yatırım Hesapları',
    'table' => 'investment_accounts',
    'cols'  => ['id','name','currency'],
    'label' => fn($r) => "{$r['name']} • {$r['currency']}",
    'order' => 'deleted_at DESC, id DESC',
  ],

  'investment_trades' => [
    'title' => '🪙 Varlık Al/Sat',
    'table' => 'investment_trades',
    'cols'  => ['id','investment_account_id','side','asset_symbol','quantity','price','fee','traded_at','note'],
    'label' => function(array $r): string {
      $side = $r['side'] === 'buy' ? 'AL' : 'SAT';
      return "{$r['traded_at']} • {$side} • {$r['asset_symbol']} • Miktar: {$r['quantity']} • Fiyat: {$r['price']}".($r['note'] ? " • {$r['note']}" : '');
    },
    'order' => 'deleted_at DESC, id DESC',
  ],
];

function safe_entity(array $ENTITY, string $e): string {
  return array_key_exists($e, $ENTITY) ? $e : 'transactions';
}

$entity = safe_entity($ENTITY, (string)($_GET['entity'] ?? 'transactions'));
$msg = '';
$err = '';

$action = (string)($_POST['action'] ?? '');
if ($action) {
  $entityPost = safe_entity($ENTITY, (string)($_POST['entity'] ?? 'transactions'));
  $table = $ENTITY[$entityPost]['table'];

  // ids can be single or multiple
  $ids = $_POST['ids'] ?? [];
  if (!is_array($ids)) $ids = [$ids];
  $ids = array_values(array_filter(array_map('intval', $ids), fn($x)=>$x>0));

  if (!$ids) {
    $err = "Lütfen en az bir kayıt seç.";
  } else {
    // build placeholders
    $ph = implode(',', array_fill(0, count($ids), '?'));
    $params = array_merge($ids, [$uid]);

    if ($action === 'restore_bulk') {
      $st = $pdo->prepare("UPDATE {$table} SET deleted_at=NULL WHERE id IN ($ph) AND user_id=?");
      $st->execute($params);
      $msg = "Seçilen kayıtlar geri alındı ✅";
    }

    if ($action === 'delete_bulk') {
      // Permanent delete
      $st = $pdo->prepare("DELETE FROM {$table} WHERE id IN ($ph) AND user_id=?");
      $st->execute($params);
      $msg = "Seçilen kayıtlar kalıcı silindi ⚠️";
    }
  }

  // stay in same entity
  $entity = $entityPost;
}

// counts for sidebar list
$counts = [];
foreach ($ENTITY as $k => $meta) {
  $st = $pdo->prepare("SELECT COUNT(*) AS c FROM {$meta['table']} WHERE user_id=? AND deleted_at IS NOT NULL");
  $st->execute([$uid]);
  $counts[$k] = (int)($st->fetch()['c'] ?? 0);
}

// fetch current entity deleted items
$meta = $ENTITY[$entity];
$cols = implode(',', $meta['cols']);
$list = $pdo->prepare("SELECT {$cols}, deleted_at FROM {$meta['table']} WHERE user_id=? AND deleted_at IS NOT NULL ORDER BY {$meta['order']} LIMIT 300");
$list->execute([$uid]);
$rows = $list->fetchAll();

require __DIR__ . '/../inc/header.php';
?>

<div class="grid grid-cols-1 xl:grid-cols-4 gap-4">

  <!-- Left: entity list -->
  <aside class="xl:col-span-1 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <h2 class="text-lg font-semibold">🗑️ Çöp Kutusu</h2>
    <p class="text-sm opacity-70 mt-1">Silinen kayıtlar burada tutulur.</p>

    <div class="mt-4 space-y-1">
      <?php foreach ($ENTITY as $k => $m): ?>
        <?php
          $active = $k === $entity;
          $badge = $counts[$k] ?? 0;
        ?>
        <a href="<?= h(page_url('trash.php')) ?>?entity=<?= h($k) ?>"
          class="flex items-center justify-between px-3 py-2 rounded-2xl border
                 <?= $active ? 'bg-slate-100 dark:bg-slate-900 border-slate-300 dark:border-slate-700 font-semibold' : 'bg-white dark:bg-slate-950 border-slate-200 dark:border-slate-800 hover:bg-slate-50 dark:hover:bg-slate-900' ?>">
          <span><?= h($m['title']) ?></span>
          <span class="text-xs px-2 py-1 rounded-full bg-slate-200 dark:bg-slate-800"><?= (int)$badge ?></span>
        </a>
      <?php endforeach; ?>
    </div>

    <div class="mt-4 text-xs opacity-70">
      Not: Bu ekran sadece <code>deleted_at IS NOT NULL</code> kayıtları gösterir.
    </div>
  </aside>

  <!-- Right: list -->
  <section class="xl:col-span-3 p-5 rounded-3xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800">
    <div class="flex flex-col md:flex-row md:items-center justify-between gap-3">
      <div>
        <h3 class="text-lg font-semibold"><?= h($meta['title']) ?></h3>
        <p class="text-sm opacity-70">Son 300 silinmiş kayıt</p>
      </div>

      <div class="text-xs opacity-70">
        Seçili: <span id="selCount">0</span>
      </div>
    </div>

    <?php if ($err): ?>
      <div class="mt-3 p-3 rounded-xl bg-rose-50 dark:bg-rose-950/40 border border-rose-200 dark:border-rose-900 text-sm">
        <?= h($err) ?>
      </div>
    <?php endif; ?>

    <?php if ($msg): ?>
      <div class="mt-3 p-3 rounded-xl bg-emerald-50 dark:bg-emerald-950/40 border border-emerald-200 dark:border-emerald-900 text-sm">
        <?= h($msg) ?>
      </div>
    <?php endif; ?>

    <form id="bulkForm" method="post" class="mt-4">
      <input type="hidden" name="entity" value="<?= h($entity) ?>">

      <div class="flex flex-wrap items-center gap-2">
        <button type="button" id="selectAllBtn"
          class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
          ✅ Tümünü Seç
        </button>

        <button type="button" id="clearBtn"
          class="px-3 py-2 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
          ✖️ Temizle
        </button>

        <button type="submit" name="action" value="restore_bulk"
          class="px-3 py-2 rounded-xl bg-primary text-white font-semibold text-sm">
          ♻️ Geri Al
        </button>

        <button type="submit" name="action" value="delete_bulk"
          class="px-3 py-2 rounded-xl bg-rose-600 text-white font-semibold text-sm"
          onclick="return confirm('Kalıcı silinsin mi? Bu işlem geri alınamaz!')">
          🗑️ Kalıcı Sil
        </button>
      </div>

      <div class="mt-4 overflow-x-auto">
        <table class="w-full text-sm">
          <thead class="text-xs uppercase opacity-70">
            <tr class="border-b border-slate-200 dark:border-slate-800">
              <th class="py-2 pr-3 text-left w-10">Seç</th>
              <th class="py-2 pr-3 text-left">Kayıt</th>
              <th class="py-2 pr-3 text-left">Silinme</th>
              <th class="py-2 pl-3 text-right w-40">İşlem</th>
            </tr>
          </thead>
          <tbody>
          <?php if (!$rows): ?>
            <tr>
              <td colspan="4" class="py-8 text-center opacity-70">Bu bölümde silinmiş kayıt yok.</td>
            </tr>
          <?php endif; ?>

          <?php foreach ($rows as $r): ?>
            <tr class="border-b border-slate-200/70 dark:border-slate-800/70">
              <td class="py-3 pr-3">
                <input class="rowChk w-4 h-4" type="checkbox" name="ids[]"
                       value="<?= (int)$r['id'] ?>">
              </td>
              <td class="py-3 pr-3">
                <div class="font-medium">
                  <?= h(($meta['label'])($r)) ?>
                </div>
                <div class="text-xs opacity-70">ID: <?= (int)$r['id'] ?></div>
              </td>
              <td class="py-3 pr-3 text-xs opacity-70 whitespace-nowrap">
                <?= h((string)$r['deleted_at']) ?>
              </td>
              <td class="py-3 pl-3 text-right whitespace-nowrap">
                <!-- single restore -->
                <button type="submit" name="action" value="restore_bulk"
                        onclick="singleSelect(<?= (int)$r['id'] ?>);"
                        class="px-3 py-1 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
                  ♻️
                </button>

                <!-- single delete -->
                <button type="submit" name="action" value="delete_bulk"
                        onclick="if(!confirm('Kalıcı silinsin mi?')) return false; singleSelect(<?= (int)$r['id'] ?>);"
                        class="px-3 py-1 rounded-xl bg-white dark:bg-slate-950 border border-slate-200 dark:border-slate-800 text-sm">
                  🗑️
                </button>
              </td>
            </tr>
          <?php endforeach; ?>
          </tbody>
        </table>
      </div>

    </form>
  </section>
</div>

<script>
  const chks = () => Array.from(document.querySelectorAll('.rowChk'));
  const selCount = document.getElementById('selCount');

  function updateCount(){
    const c = chks().filter(x=>x.checked).length;
    selCount.textContent = String(c);
  }

  document.addEventListener('change', (e)=>{
    if (e.target.classList.contains('rowChk')) updateCount();
  });

  document.getElementById('selectAllBtn')?.addEventListener('click', ()=>{
    chks().forEach(x=>x.checked = true);
    updateCount();
  });

  document.getElementById('clearBtn')?.addEventListener('click', ()=>{
    chks().forEach(x=>x.checked = false);
    updateCount();
  });

  // used for single action buttons
  function singleSelect(id){
    chks().forEach(x=>x.checked = (parseInt(x.value,10) === id));
    updateCount();
  }
  window.singleSelect = singleSelect;

  updateCount();
</script>

<?php require __DIR__ . '/../inc/footer.php'; ?>
