<?php
declare(strict_types=1);

require_once __DIR__ . '/db.php';
require_once __DIR__ . '/security.php';

function auth_user(): ?array {
  start_session();
  return $_SESSION['user'] ?? null;
}

function require_auth(): array {
  $u = auth_user();
  if (!$u) {
    header('Location: /login.php');
    exit;
  }
  return $u;
}

function require_admin(): array {
  $u = require_auth();
  if (($u['role'] ?? '') !== 'admin') {
    http_response_code(403);
    exit('Yetkisiz.');
  }
  return $u;
}

function login_attempt(string $username, string $password): array {
  $pdo = db();
  $st = $pdo->prepare('SELECT id, username, full_name, role, password_hash, is_active FROM users WHERE username = ? LIMIT 1');
  $st->execute([$username]);
  $u = $st->fetch();

  if (!$u || !(int)$u['is_active']) {
    return ['ok' => false, 'error' => 'Kullanıcı bulunamadı veya pasif.'];
  }
  if (!password_verify($password, (string)$u['password_hash'])) {
    return ['ok' => false, 'error' => 'Şifre hatalı.'];
  }

  start_session();
  session_regenerate_id(true);
  $_SESSION['user'] = [
    'id' => (int)$u['id'],
    'username' => (string)$u['username'],
    'full_name' => (string)$u['full_name'],
    'role' => (string)$u['role'],
  ];
  return ['ok' => true];
}

function logout(): void {
  start_session();
  $_SESSION = [];
  if (ini_get('session.use_cookies')) {
    $p = session_get_cookie_params();
    setcookie(session_name(), '', time()-42000, $p['path'] ?? '/', $p['domain'] ?? '', (bool)($p['secure'] ?? false), true);
  }
  session_destroy();
}
