<?php
declare(strict_types=1);

function start_session(): void {
  $cfg = require __DIR__ . '/config.php';

  // IMPORTANT: session cookie params can only be set BEFORE session_start().
  // start_session() is called from multiple places (e.g. csrf_token + csrf_verify),
  // so it MUST be idempotent.
  if (session_status() !== PHP_SESSION_ACTIVE) {
    $cookieParams = session_get_cookie_params();
    $secure = (bool)$cfg['is_production'];

    session_set_cookie_params([
      'lifetime' => 0,
      'path' => $cookieParams['path'] ?? '/',
      'domain' => $cookieParams['domain'] ?? '',
      'secure' => $secure,
      'httponly' => true,
      'samesite' => 'Lax',
    ]);

    session_start();
  }

  // TTL
  $ttl = (int)$cfg['session_ttl'];
  $now = time();
  if (!isset($_SESSION['_last'])) {
    $_SESSION['_last'] = $now;
  } else if (($now - (int)$_SESSION['_last']) > $ttl) {
    session_unset();
    session_destroy();
    session_start();
  }
  $_SESSION['_last'] = $now;
}

function csrf_token(): string {
  start_session();
  if (empty($_SESSION['_csrf'])) {
    $_SESSION['_csrf'] = bin2hex(random_bytes(16));
  }
  return (string)$_SESSION['_csrf'];
}

function csrf_verify(): void {
  start_session();
  $token = $_POST['_csrf'] ?? '';
  if (!is_string($token) || !$token || !hash_equals($_SESSION['_csrf'] ?? '', $token)) {
    http_response_code(419);
    exit('CSRF doğrulaması başarısız.');
  }
}

function e(string $s): string {
  return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}
