<?php
declare(strict_types=1);

require __DIR__ . '/../app/helpers.php';

session_start_safe();

$items = get_items();
$flash = $_SESSION['flash'] ?? null;
unset($_SESSION['flash']);

function flash(string $msg, string $type = 'ok'): void {
  $_SESSION['flash'] = ['msg' => $msg, 'type' => $type];
}

$id = (string)($_GET['id'] ?? '');
if ($id === '') {
  flash('Düzenlenecek kayıt bulunamadı.', 'err');
  redirect('records.php');
}

$index = null;
foreach ($items as $i => $it) {
  if ((string)($it['id'] ?? '') === $id) {
    $index = $i;
    break;
  }
}

if ($index === null) {
  flash('Kayıt bulunamadı.', 'err');
  redirect('records.php');
}

$item = $items[$index];

if ($_SERVER['REQUEST_METHOD'] === 'POST' && (string)($_POST['action'] ?? '') === 'update_item') {
  csrf_or_fail();

  $name = trim((string)($_POST['name'] ?? ''));
  $priceRaw = trim((string)($_POST['price'] ?? ''));
  $barcode = trim((string)($_POST['barcode'] ?? ''));
  $unit = trim((string)($_POST['unit'] ?? ''));

  if ($name === '') {
    flash('Stok adı zorunlu.', 'err');
    redirect('edit.php?id=' . urlencode($id));
  }

  $price = '';
  if ($priceRaw !== '') {
    $normalized = str_replace(',', '.', $priceRaw);
    if (!is_numeric($normalized)) {
      flash('Fiyat sayısal olmalı (örn: 12.50).', 'err');
      redirect('edit.php?id=' . urlencode($id));
    }
    $price = (string)$normalized;
  }

  $items[$index]['name'] = $name;
  $items[$index]['price'] = $price;
  $items[$index]['barcode'] = $barcode;
  $items[$index]['unit'] = $unit;
  $items[$index]['updated_at'] = now_tr();

  save_items($items);
  flash('Kayıt güncellendi.', 'ok');
  redirect('records.php');
}

?><!doctype html>
<html lang="tr" class="h-full">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Düzenle — Stok Kodu</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script>tailwind.config = { darkMode: 'class' };</script>
</head>
<body class="h-full bg-gray-50 text-gray-900">
  <div class="mx-auto max-w-3xl p-4 sm:p-6">
    <div class="flex items-center justify-between gap-3">
      <div>
        <h1 class="text-xl sm:text-2xl font-semibold">Kayıt Düzenle</h1>
        <p class="text-sm text-gray-600">Stok kodu sabit kalır, alanları güncelleyebilirsin.</p>
      </div>
      <div class="flex items-center gap-2">
        <a href="records.php" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">← Kayıtlar</a>
        <a href="index.php" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Oluşturucu</a>
        <button id="themeBtn" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Tema</button>
      </div>
    </div>

    <?php if (is_array($flash)): ?>
      <div class="mt-4 rounded-2xl border px-4 py-3 text-sm <?php echo ($flash['type'] ?? '') === 'err' ? 'border-red-200 bg-red-50 text-red-800' : 'border-emerald-200 bg-emerald-50 text-emerald-900'; ?>">
        <?php echo e((string)($flash['msg'] ?? '')); ?>
      </div>
    <?php endif; ?>

    <div class="mt-6 rounded-3xl border border-gray-200 bg-white p-4 sm:p-6">
      <div class="rounded-2xl border border-gray-200 bg-gray-50 p-3">
        <div class="text-xs text-gray-500">Stok Kodu</div>
        <div class="mt-1 font-mono text-base sm:text-lg" id="codeText"><?php echo e((string)($item['code'] ?? '')); ?></div>
        <button class="mt-2 rounded-xl border border-gray-200 bg-white px-3 py-1.5 text-xs hover:bg-gray-50" id="copyCode" type="button">Kodu kopyala</button>
      </div>

      <form class="mt-5 grid gap-4" method="post">
        <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>" />
        <input type="hidden" name="action" value="update_item" />

        <div class="grid gap-4 sm:grid-cols-2">
          <label class="grid gap-1 sm:col-span-2">
            <span class="text-sm font-medium">Stok Adı</span>
            <input name="name" required class="rounded-2xl border border-gray-200 bg-white px-3 py-2" value="<?php echo e((string)($item['name'] ?? '')); ?>" />
          </label>

          <label class="grid gap-1">
            <span class="text-sm font-medium">Fiyat</span>
            <input name="price" inputmode="decimal" class="rounded-2xl border border-gray-200 bg-white px-3 py-2" value="<?php echo e((string)($item['price'] ?? '')); ?>" />
          </label>

          <label class="grid gap-1">
            <span class="text-sm font-medium">Birim</span>
            <input name="unit" class="rounded-2xl border border-gray-200 bg-white px-3 py-2" value="<?php echo e((string)($item['unit'] ?? '')); ?>" />
          </label>

          <label class="grid gap-1 sm:col-span-2">
            <span class="text-sm font-medium">Barkod</span>
            <input name="barcode" class="rounded-2xl border border-gray-200 bg-white px-3 py-2" value="<?php echo e((string)($item['barcode'] ?? '')); ?>" />
          </label>
        </div>

        <div class="flex flex-wrap items-center gap-2 pt-1">
          <button class="rounded-2xl bg-gray-900 px-4 py-2 text-sm font-semibold text-white hover:bg-black">Kaydet</button>
          <a href="records.php" class="rounded-2xl border border-gray-200 bg-white px-4 py-2 text-sm hover:bg-gray-50">Vazgeç</a>
        </div>

        <div class="mt-2 text-xs text-gray-500">
          <?php if (!empty($item['updated_at'])): ?>
            Son güncelleme: <?php echo e((string)$item['updated_at']); ?>
          <?php endif; ?>
        </div>
      </form>
    </div>
  </div>

  <script src="assets/app.js"></script>
  <script>
    (function(){
      const btn = document.getElementById('copyCode');
      const txt = document.getElementById('codeText');
      if (!btn || !txt) return;
      btn.addEventListener('click', async () => {
        try {
          await navigator.clipboard.writeText(txt.textContent.trim());
          btn.textContent = 'Kopyalandı';
          setTimeout(() => btn.textContent = 'Kodu kopyala', 1200);
        } catch(e) {}
      });
    })();
  </script>
</body>
</html>
