<?php
declare(strict_types=1);

require __DIR__ . '/../app/helpers.php';

session_start_safe();

function flash(string $msg, string $type = 'ok'): void {
  $_SESSION['flash'] = ['msg' => $msg, 'type' => $type];
}

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
  redirect('records.php');
}

csrf_or_fail();

if (!isset($_FILES['csv']) || !is_array($_FILES['csv'])) {
  flash('CSV dosyasi secilmedi.', 'err');
  redirect('records.php');
}

$file = $_FILES['csv'];
if (($file['error'] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) {
  flash('CSV yuklenemedi. Hata kodu: ' . (string)($file['error'] ?? ''), 'err');
  redirect('records.php');
}

$tmp = (string)($file['tmp_name'] ?? '');
if ($tmp === '' || !is_uploaded_file($tmp)) {
  flash('CSV yukleme gecersiz.', 'err');
  redirect('records.php');
}

$raw = file_get_contents($tmp);
if ($raw === false) {
  flash('CSV okunamadi.', 'err');
  redirect('records.php');
}

// UTF-8 BOM temizle
$raw = preg_replace('/^\xEF\xBB\xBF/', '', $raw);
$lines = preg_split("/\r\n|\n|\r/", $raw);
$lines = array_values(array_filter($lines, fn($l) => trim((string)$l) !== ''));
if (!$lines) {
  flash('CSV bos gorunuyor.', 'err');
  redirect('records.php');
}

$first = (string)$lines[0];
$delim = (substr_count($first, ';') >= substr_count($first, ',')) ? ';' : ',';

$header = str_getcsv(array_shift($lines), $delim);
$header = array_map(fn($h) => mb_strtolower(trim((string)$h)), $header);

$wanted = [
  'stok kodu' => 'code',
  'stok adi' => 'name',
  'stok adı' => 'name',
  'isim' => 'name',
  'fiyat' => 'price',
  'barkod' => 'barcode',
  'birim' => 'unit',
  'biri̇m' => 'unit'
];

$map = [];
foreach ($header as $idx => $col) {
  $col = trim($col);
  if (isset($wanted[$col])) $map[(int)$idx] = $wanted[$col];
}

if (!in_array('code', $map, true)) {
  flash('CSV basliklarinda "Stok Kodu" kolonu bulunamadi.', 'err');
  redirect('records.php');
}

$rows = [];
foreach ($lines as $line) {
  $cells = str_getcsv((string)$line, $delim);
  $row = ['code'=>'','name'=>'','price'=>'','barcode'=>'','unit'=>''];
  foreach ($map as $idx => $key) {
    $val = trim((string)($cells[$idx] ?? ''));
    if ($key === 'price') {
      // v5 export fiyat string; burada da string tutuyoruz
      $val = str_replace(',', '.', $val);
    }
    $row[$key] = $val;
  }
  if ($row['code'] !== '') $rows[] = $row;
}

if (!$rows) {
  flash('CSV icinde aktarilacak satir bulunamadi.', 'err');
  redirect('records.php');
}

$items = get_items();
$byCode = [];
foreach ($items as $i => $it) {
  $c = (string)($it['code'] ?? '');
  if ($c !== '') $byCode[$c] = $i;
}

$added = 0;
$updated = 0;
foreach ($rows as $r) {
  $code = (string)$r['code'];
  if (isset($byCode[$code])) {
    $idx = $byCode[$code];
    $items[$idx]['name'] = $r['name'] !== '' ? $r['name'] : (string)($items[$idx]['name'] ?? '');
    $items[$idx]['price'] = $r['price'] !== '' ? $r['price'] : (string)($items[$idx]['price'] ?? '');
    $items[$idx]['barcode'] = $r['barcode'] !== '' ? $r['barcode'] : (string)($items[$idx]['barcode'] ?? '');
    $items[$idx]['unit'] = $r['unit'] !== '' ? $r['unit'] : (string)($items[$idx]['unit'] ?? '');
    $updated++;
  } else {
    $items[] = [
      'id' => bin2hex(random_bytes(8)),
      'code' => $code,
      'name' => $r['name'],
      'price' => $r['price'],
      'barcode' => $r['barcode'],
      'unit' => $r['unit'],
      'created_at' => now_tr()
    ];
    $byCode[$code] = count($items) - 1;
    $added++;
  }
}

save_items(array_values($items));

flash("Iceri aktarim tamamlandi: $added eklendi, $updated guncellendi.", 'ok');
redirect('records.php');
