<?php
declare(strict_types=1);

require __DIR__ . '/../app/helpers.php';

session_start_safe();

$lists = get_lists();
$flash = $_SESSION['flash'] ?? null;
unset($_SESSION['flash']);

function flash(string $msg, string $type = 'ok'): void {
  $_SESSION['flash'] = ['msg' => $msg, 'type' => $type];
}

$validKeys = ['categories', 'types', 'groups', 'reyon'];

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $action = (string)($_POST['action'] ?? '');

  if ($action === 'add') {
    csrf_or_fail();

    $key = (string)($_POST['list_key'] ?? '');
    $code = trim((string)($_POST['code'] ?? ''));
    $label = trim((string)($_POST['label'] ?? ''));

    if (!in_array($key, $validKeys, true)) {
      flash('Geçersiz liste.', 'err');
      redirect('lists.php');
    }

    if ($code === '' || $label === '') {
      flash('Kod ve açıklama zorunlu.', 'err');
      redirect('lists.php');
    }

    // Normalize numeric codes to 3 digits for category/group/reyon.
    if (in_array($key, ['categories', 'groups', 'reyon'], true)) {
      $code = pad_code($code, 3);
    }

    // Types typically 2 chars, but keep as-is if user wants.

    foreach ($lists[$key] as $row) {
      if ((string)($row['code'] ?? '') === $code) {
        flash('Bu kod zaten var.', 'err');
        redirect('lists.php');
      }
    }

    $lists[$key][] = ['code' => $code, 'label' => $label];
    save_lists($lists);
    flash('Eklendi: ' . $code, 'ok');
    redirect('lists.php');
  }

  if ($action === 'reset_items') {
    csrf_or_fail();
    save_items([]);
    flash('Kayıtlar silindi.', 'ok');
    redirect('lists.php');
  }

  if ($action === 'reset_counters') {
    csrf_or_fail();
    save_counters([]);
    flash('Sayaçlar sıfırlandı (sıra numarası yeniden 000001’den başlar).', 'ok');
    redirect('lists.php');
  }
}

if (($_GET['action'] ?? '') === 'delete') {
  csrf_or_fail();
  $key = (string)($_GET['list_key'] ?? '');
  $code = (string)($_GET['code'] ?? '');

  if (in_array($key, $validKeys, true)) {
    $lists[$key] = array_values(array_filter($lists[$key], fn($r) => (string)($r['code'] ?? '') !== $code));
    save_lists($lists);
    flash('Silindi: ' . $code, 'ok');
  }
  redirect('lists.php');
}

?><!doctype html>
<html lang="tr" class="h-full">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Listeleri Yönet</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script>tailwind.config = { darkMode: 'class' };</script>
</head>
<body class="h-full bg-gray-50 text-gray-900">
  <div class="mx-auto max-w-5xl p-4 sm:p-6">
    <div class="flex items-center justify-between gap-3">
      <div>
        <h1 class="text-xl sm:text-2xl font-semibold">Listeleri Yönet</h1>
        <p class="text-sm text-gray-600">Kategori / Tip / Grup / Reyon listesini buradan genişlet.</p>
      </div>
      <div class="flex items-center gap-2">
        <a href="index.php" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">← Geri</a>
        <a href="records.php" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Kayıtlar</a>
        <button id="themeBtn" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Tema</button>
      </div>
    </div>

    <?php if (is_array($flash)): ?>
      <div class="mt-4 rounded-2xl border px-4 py-3 text-sm <?php echo ($flash['type'] ?? '') === 'err' ? 'border-red-200 bg-red-50 text-red-800' : 'border-emerald-200 bg-emerald-50 text-emerald-900'; ?>">
        <?php echo e((string)($flash['msg'] ?? '')); ?>
      </div>
    <?php endif; ?>

    <div class="mt-6 grid gap-4 lg:grid-cols-2">
      <div class="rounded-3xl border border-gray-200 bg-white p-4 sm:p-6">
        <h2 class="text-base font-semibold">Yeni değer ekle</h2>
        <p class="mt-1 text-sm text-gray-600">Kod + açıklama gir. Kategori/Grup/Reyon sayısal ise otomatik 3 hane yapılır.</p>

        <form class="mt-4 grid gap-3" method="post">
          <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>" />
          <input type="hidden" name="action" value="add" />

          <label class="grid gap-1">
            <span class="text-sm font-medium">Liste</span>
            <select name="list_key" class="rounded-2xl border border-gray-200 bg-white px-3 py-2">
              <option value="categories">Kategori</option>
              <option value="types">Ürün Tipi</option>
              <option value="groups">Grup</option>
              <option value="reyon">Reyon</option>
            </select>
          </label>

          <label class="grid gap-1">
            <span class="text-sm font-medium">Kod</span>
            <input name="code" class="rounded-2xl border border-gray-200 bg-white px-3 py-2" placeholder="007 veya AS" />
          </label>

          <label class="grid gap-1">
            <span class="text-sm font-medium">Açıklama</span>
            <input name="label" class="rounded-2xl border border-gray-200 bg-white px-3 py-2" placeholder="Örn: PAKETLİ / Al-Sat" />
          </label>

          <button class="rounded-2xl bg-gray-900 px-4 py-2 text-sm font-semibold text-white hover:bg-black">Ekle</button>
        </form>

        <div class="mt-6 rounded-2xl border border-gray-200 bg-gray-50 p-4">
          <div class="text-sm font-semibold">Bakım</div>
          <div class="mt-3 flex flex-wrap gap-2">
            <form method="post" onsubmit="return confirm('Tüm kayıtlar silinsin mi?')">
              <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>" />
              <input type="hidden" name="action" value="reset_items" />
              <button class="rounded-2xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Kayıtları Sil</button>
            </form>
            <form method="post" onsubmit="return confirm('Sayaçlar sıfırlansın mı?')">
              <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>" />
              <input type="hidden" name="action" value="reset_counters" />
              <button class="rounded-2xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Sayaçları Sıfırla</button>
            </form>
          </div>
          <p class="mt-2 text-xs text-gray-600">Sayaçları sıfırlarsan aynı kombinasyon için sıra numarası yeniden 000001’den başlar.</p>
        </div>
      </div>

      <div class="rounded-3xl border border-gray-200 bg-white p-4 sm:p-6">
        <h2 class="text-base font-semibold">Mevcut listeler</h2>
        <p class="mt-1 text-sm text-gray-600">Silmek için çöp ikonuna tıkla.</p>

        <?php
          $labels = [
            'categories' => 'Kategori',
            'types' => 'Ürün Tipi',
            'groups' => 'Grup',
            'reyon' => 'Reyon',
          ];
        ?>

        <div class="mt-4 space-y-4">
          <?php foreach ($validKeys as $k): ?>
            <div class="rounded-2xl border border-gray-200">
              <div class="flex items-center justify-between px-3 py-2">
                <div class="text-sm font-semibold"><?php echo e($labels[$k]); ?></div>
                <div class="text-xs text-gray-500"><?php echo count($lists[$k]); ?> adet</div>
              </div>
              <div class="border-t max-h-[220px] overflow-auto">
                <?php if (count($lists[$k]) === 0): ?>
                  <div class="p-3 text-sm text-gray-500">Boş</div>
                <?php else: ?>
                  <?php foreach ($lists[$k] as $row): $code=(string)($row['code']??''); $label=(string)($row['label']??''); ?>
                    <div class="flex items-center justify-between gap-2 px-3 py-2 hover:bg-gray-50">
                      <div>
                        <div class="font-mono text-sm"><?php echo e($code); ?></div>
                        <div class="text-xs text-gray-600"><?php echo e($label); ?></div>
                      </div>
                      <a class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-xs hover:bg-gray-50" href="?action=delete&list_key=<?php echo e($k); ?>&code=<?php echo e($code); ?>&csrf=<?php echo e(csrf_token()); ?>" onclick="return confirm('Silinsin mi?')">Sil</a>
                    </div>
                  <?php endforeach; ?>
                <?php endif; ?>
              </div>
            </div>
          <?php endforeach; ?>
        </div>
      </div>
    </div>

    <footer class="mt-8 text-xs text-gray-500">
      <div>Liste dosyası: <span class="font-mono">data/lists.json</span></div>
    </footer>
  </div>

  <script src="assets/app.js"></script>
</body>
</html>
