<?php
declare(strict_types=1);

require __DIR__ . '/../app/helpers.php';

session_start_safe();

$lists = get_lists();
$counters = get_counters();
$items = get_items();

$flash = $_SESSION['flash'] ?? null;
unset($_SESSION['flash']);

function flash(string $msg, string $type='ok'): void {
  $_SESSION['flash'] = ['msg' => $msg, 'type' => $type];
}

function build_counter_key(string $cat, string $type, string $group, string $reyon): string {
  return $cat . '|' . $type . '|' . $group . '|' . $reyon;
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && ($_POST['action'] ?? '') === 'generate') {
  csrf_or_fail();

  $name = trim((string)($_POST['name'] ?? ''));
  $priceRaw = trim((string)($_POST['price'] ?? ''));
  $barcode = trim((string)($_POST['barcode'] ?? ''));
  $unit = trim((string)($_POST['unit'] ?? ''));

  $cat   = pad_code((string)($_POST['category'] ?? ''), 3);
  $type  = pad_code((string)($_POST['type'] ?? ''), 2);
  $group = pad_code((string)($_POST['group'] ?? ''), 3);
  $reyon = pad_code((string)($_POST['reyon'] ?? ''), 3);

  if ($name === '') {
    flash('Stok adı (ürün ismi) zorunlu.', 'err');
    redirect('index.php');
  }

  // price is optional; if provided validate basic numeric
  $price = '';
  if ($priceRaw !== '') {
    $normalized = str_replace(',', '.', $priceRaw);
    if (!is_numeric($normalized)) {
      flash('Fiyat sayısal olmalı (örn: 12.50).', 'err');
      redirect('index.php');
    }
    // Keep as string with dot, trim trailing zeros nicely
    $price = (string)$normalized;
  }

  if ($cat === '' || $type === '' || $group === '' || $reyon === '') {
    flash('Lütfen tüm alanları seç.', 'err');
    redirect('index.php');
  }

  $key = build_counter_key($cat, $type, $group, $reyon);
  $next = (int)($counters[$key] ?? 0) + 1;
  $counters[$key] = $next;
  save_counters($counters);

  $serial = str_pad((string)$next, 6, '0', STR_PAD_LEFT);
  $stockCode = $cat . '.' . $type . '.' . $group . '.' . $reyon . '.' . $serial;

  $item = [
    'id' => bin2hex(random_bytes(8)),
    'code' => $stockCode,
    'name' => $name,
    'price' => $price,
    'barcode' => $barcode,
    'unit' => $unit,
    'category' => ['code' => $cat, 'label' => find_label($lists['categories'], $cat)],
    'type'     => ['code' => $type, 'label' => find_label($lists['types'], $type)],
    'group'    => ['code' => $group, 'label' => find_label($lists['groups'], $group)],
    'reyon'    => ['code' => $reyon, 'label' => find_label($lists['reyon'], $reyon)],
    'serial' => $serial,
    'created_at' => now_tr(),
  ];

  array_unshift($items, $item);
  // Keep last 500 for performance.
  if (count($items) > 500) {
    $items = array_slice($items, 0, 500);
  }
  save_items($items);

  flash('Stok kodu oluşturuldu: ' . $stockCode, 'ok');
  redirect('index.php');
}

$selected = [
  'category' => (string)($_GET['category'] ?? ($lists['categories'][0]['code'] ?? '')),
  'type'     => (string)($_GET['type'] ?? ($lists['types'][0]['code'] ?? '')),
  'group'    => (string)($_GET['group'] ?? ($lists['groups'][0]['code'] ?? '')),
  'reyon'    => (string)($_GET['reyon'] ?? ($lists['reyon'][0]['code'] ?? '')),
];

$cat   = pad_code($selected['category'], 3);
$type  = pad_code($selected['type'], 2);
$group = pad_code($selected['group'], 3);
$reyon = pad_code($selected['reyon'], 3);

$previewKey = build_counter_key($cat, $type, $group, $reyon);
$previewNext = (int)($counters[$previewKey] ?? 0) + 1;
$previewSerial = str_pad((string)$previewNext, 6, '0', STR_PAD_LEFT);
$previewCode = ($cat && $type && $group && $reyon)
  ? ($cat . '.' . $type . '.' . $group . '.' . $reyon . '.' . $previewSerial)
  : '---';

?><!doctype html>
<html lang="tr" class="h-full">
<head>
  <meta charset="utf-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <title>Stok Kodu Oluşturucu</title>
  <script src="https://cdn.tailwindcss.com"></script>
  <script>
    tailwind.config = { darkMode: 'class' };
  </script>
</head>
<body class="h-full bg-gray-50 text-gray-900">
  <div class="mx-auto max-w-5xl p-4 sm:p-6">
    <div class="flex items-center justify-between gap-3">
      <div>
        <h1 class="text-xl sm:text-2xl font-semibold">Stok Kodu Oluşturucu</h1>
        <p class="text-sm text-gray-600">Veritabanı yok: JSON ile kaydeder. Format: <span class="font-mono">007.AS.501.009.000018</span></p>
      </div>
      <div class="flex items-center gap-2">
        <a href="lists.php" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Listeleri Yönet</a>
        <a href="records.php" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Kayıtlar</a>
        <button id="themeBtn" class="rounded-xl border border-gray-200 bg-white px-3 py-2 text-sm hover:bg-gray-50">Tema</button>
      </div>
    </div>

    <?php if (is_array($flash)): ?>
      <div class="mt-4 rounded-2xl border px-4 py-3 text-sm <?php echo ($flash['type'] ?? '') === 'err' ? 'border-red-200 bg-red-50 text-red-800' : 'border-emerald-200 bg-emerald-50 text-emerald-900'; ?>">
        <?php echo e((string)($flash['msg'] ?? '')); ?>
      </div>
    <?php endif; ?>

    <div class="mt-6 grid gap-4 lg:grid-cols-3">
      <div class="lg:col-span-2 rounded-3xl border border-gray-200 bg-white p-4 sm:p-6">
        <div class="flex items-start justify-between gap-3">
          <div>
            <h2 class="text-base font-semibold">Yeni stok kodu</h2>
            <p class="text-sm text-gray-600">Kod + yazı birlikte görünür. Kodlar senin sıranla gider.</p>
          </div>
          <div class="text-right">
            <div class="text-xs text-gray-500">Önizleme (son parça otomatik artar)</div>
            <div class="mt-1 font-mono text-sm sm:text-base" id="previewCode"><?php echo e($previewCode); ?></div>
          </div>
        </div>

        <form class="mt-5 grid gap-4" method="post" action="index.php">
          <input type="hidden" name="csrf" value="<?php echo e(csrf_token()); ?>" />
          <input type="hidden" name="action" value="generate" />

          <div class="grid gap-4 sm:grid-cols-2">
            <label class="grid gap-1">
              <span class="text-sm font-medium">Stok Adı</span>
              <input name="name" required class="rounded-2xl border border-gray-200 bg-white px-3 py-2" placeholder="Örn: Ürün Adı" />
            </label>

            <label class="grid gap-1">
              <span class="text-sm font-medium">Birim</span>
              <input name="unit" class="rounded-2xl border border-gray-200 bg-white px-3 py-2" placeholder="Örn: Adet / Kg / Lt" />
            </label>

            <label class="grid gap-1">
              <span class="text-sm font-medium">Fiyat</span>
              <input name="price" inputmode="decimal" class="rounded-2xl border border-gray-200 bg-white px-3 py-2" placeholder="Örn: 12.50" />
            </label>

            <label class="grid gap-1">
              <span class="text-sm font-medium">Barkod</span>
              <input name="barcode" class="rounded-2xl border border-gray-200 bg-white px-3 py-2" placeholder="EAN/UPC" />
            </label>
          </div>

          <div class="grid gap-4 sm:grid-cols-2">
            <label class="grid gap-1">
              <span class="text-sm font-medium">Kategori</span>
              <select name="category" id="category" class="rounded-2xl border border-gray-200 bg-white px-3 py-2">
                <?php foreach ($lists['categories'] as $row): $code=(string)($row['code']??''); $label=(string)($row['label']??''); ?>
                  <option value="<?php echo e($code); ?>" <?php echo $code === $selected['category'] ? 'selected' : ''; ?>><?php echo e($code . ' — ' . $label); ?></option>
                <?php endforeach; ?>
              </select>
            </label>

            <label class="grid gap-1">
              <span class="text-sm font-medium">Ürün Tipi</span>
              <select name="type" id="type" class="rounded-2xl border border-gray-200 bg-white px-3 py-2">
                <?php foreach ($lists['types'] as $row): $code=(string)($row['code']??''); $label=(string)($row['label']??''); ?>
                  <option value="<?php echo e($code); ?>" <?php echo $code === $selected['type'] ? 'selected' : ''; ?>><?php echo e($code . ' — ' . $label); ?></option>
                <?php endforeach; ?>
              </select>
            </label>

            <label class="grid gap-1">
              <span class="text-sm font-medium">Grup</span>
              <select name="group" id="group" class="rounded-2xl border border-gray-200 bg-white px-3 py-2">
                <?php foreach ($lists['groups'] as $row): $code=(string)($row['code']??''); $label=(string)($row['label']??''); ?>
                  <option value="<?php echo e($code); ?>" <?php echo $code === $selected['group'] ? 'selected' : ''; ?>><?php echo e($code . ' — ' . $label); ?></option>
                <?php endforeach; ?>
              </select>
            </label>

            <label class="grid gap-1">
              <span class="text-sm font-medium">Reyon</span>
              <select name="reyon" id="reyon" class="rounded-2xl border border-gray-200 bg-white px-3 py-2">
                <?php foreach ($lists['reyon'] as $row): $code=(string)($row['code']??''); $label=(string)($row['label']??''); ?>
                  <option value="<?php echo e($code); ?>" <?php echo $code === $selected['reyon'] ? 'selected' : ''; ?>><?php echo e($code . ' — ' . $label); ?></option>
                <?php endforeach; ?>
              </select>
            </label>
          </div>

          <div class="flex flex-wrap items-center gap-2 pt-1">
            <button class="rounded-2xl bg-gray-900 px-4 py-2 text-sm font-semibold text-white hover:bg-black">Stok Kodu Oluştur</button>
            <button type="button" id="copyPreview" class="rounded-2xl border border-gray-200 bg-white px-4 py-2 text-sm hover:bg-gray-50">Önizlemeyi Kopyala</button>
          </div>

          <div class="rounded-2xl border border-gray-200 bg-gray-50 p-3 text-sm">
            <div class="font-semibold">Açıklama</div>
            <div class="mt-1 text-gray-700">
              <span class="font-mono">007</span> = Kategori, <span class="font-mono">AS</span> = Tip, <span class="font-mono">501</span> = Grup, <span class="font-mono">009</span> = Reyon, <span class="font-mono">000018</span> = Sıra (otomatik)
            </div>
          </div>
        </form>
      </div>

      <div class="rounded-3xl border border-gray-200 bg-white p-4 sm:p-6">
        <h3 class="text-base font-semibold">Son oluşturulanlar</h3>
        <p class="mt-1 text-sm text-gray-600">Tıkla kopyala. En fazla 500 kayıt tutulur.</p>

        <div class="mt-4 space-y-2 max-h-[480px] overflow-auto pr-1">
          <?php foreach (array_slice($items, 0, 15) as $it): ?>
            <button class="w-full text-left rounded-2xl border border-gray-200 bg-white px-3 py-2 hover:bg-gray-50" data-copy="<?php echo e((string)($it['code'] ?? '')); ?>">
              <div class="flex items-center justify-between gap-2">
                <div class="font-mono text-sm"><?php echo e((string)($it['code'] ?? '')); ?></div>
                <div class="text-xs text-gray-500"><?php echo e((string)($it['created_at'] ?? '')); ?></div>
              </div>
              <div class="mt-1 text-xs text-gray-700 font-medium">
                <?php echo e((string)($it['name'] ?? '')); ?>
              </div>
              <div class="mt-1 text-xs text-gray-600">
                <?php echo e(($it['price'] ?? '') !== '' ? ('Fiyat: ' . (string)$it['price']) : ''); ?>
                <?php echo e(($it['unit'] ?? '') !== '' ? (' · Birim: ' . (string)$it['unit']) : ''); ?>
                <?php echo e(($it['barcode'] ?? '') !== '' ? (' · Barkod: ' . (string)$it['barcode']) : ''); ?>
              </div>
              <div class="mt-1 text-xs text-gray-600">
                <?php echo e(($it['category']['label'] ?? '') ?: '—'); ?> · <?php echo e(($it['type']['label'] ?? '') ?: '—'); ?>
              </div>
            </button>
          <?php endforeach; ?>
          <?php if (count($items) === 0): ?>
            <div class="text-sm text-gray-500">Henüz kayıt yok.</div>
          <?php endif; ?>
        </div>

        <div class="mt-4">
          <a href="records.php" class="text-sm underline text-gray-700">Tüm kayıtları gör →</a>
        </div>
      </div>
    </div>

    <footer class="mt-8 text-xs text-gray-500">
      <div>Dosyalar: <span class="font-mono">data/lists.json</span>, <span class="font-mono">data/counters.json</span>, <span class="font-mono">data/items.json</span></div>
    </footer>
  </div>

  <script src="assets/app.js"></script>
</body>
</html>
